﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief    Private header file for debugging interfaces
 */

#pragma once

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>

#ifdef NN_BUILD_CONFIG_OS_WIN
#error 'Debug util doesn't support Windows, make sure to include this header only for the Horizon environment
#endif

namespace nn { namespace ssl {

class Debug
{
public:
    // ------------------------------------------------------------------------
    // Structures / Enums for data
    // ------------------------------------------------------------------------

    enum PeriodicDumpType
    {
        PeriodicDumpType_HeapStats,
        PeriodicDumpType_SessionCacheInfo,
    };

    typedef struct PeriodicDumpConfig
    {
        bool             enable;
        PeriodicDumpType type;
    } PeriodicDumpConfig;

    typedef struct HeapStats
    {
        size_t curSize;
        size_t minSize;
        size_t maxSize;
    } HeapStats;

    typedef struct TrackStats
    {
        uint64_t elapsedMsec;
        int64_t  delta;
        size_t   curHeapSpace;
    } TrackStats;

    typedef struct HeapTrackPoint
    {
        static const size_t NameSize = 17;
        HeapTrackPoint(); // Constructor

        char _name[NameSize];
    } HeapTrackPoint;

    // ------------------------------------------------------------------------
    // Structures / Enums for Interfaces
    // ------------------------------------------------------------------------
    typedef struct Input
    {
        const char* pBuffer;
        size_t bufferSize;
    } Input;

    typedef struct Output
    {
        char* pBuffer;
        size_t bufferSize;
    } Output;

    enum IoctlCommand
    {
        IoctlCommand_Invalid = 0x00,        //!< Invalid command
        IoctlCommand_ConfigurePeriodicDump, //!< Enable/Disable periodic heap stats dump by the SSL process
        IoctlCommand_GetHeapStats,          //!< Get the SSL process heap stats
        IoctlCommand_StartHeapTrack,        //!< Start heap track
        IoctlCommand_EndHeapTrack,          //!< End heap track
    };

    /**
     * @brief      Send command to the SSL process to process debugging information.
     *
     * @param[out] pOutBufferInfo Pointer to the buffer in which output data is written
     * @param[in]  pInBufferInfo  Pointer to the buffer in which input data is stored
     * @param[in]  cmd            Command to process
     *
     * @return     Result of the process is returned
     *
     * @details    This API sends command to the SSL process to handle necessary process for
     *             debugging the SSL process. Command to process is specified in cmd.
     *             pOutBufferInfo is the pointer to the buffer where the result of the process
     *             gets stored. pInBufferInfo is the pointer to the buffer where necesarry
     *             information for the command is stored.
     *             The size of both pInBufferInfo and pInBufferInfo need to be configured properly
     *             for the specific command. More details are found below.
     *
     *             ### Supported commands
     *             [IoctlCommand_ConfigurePeriodicDump]
     *             - Input : PeriodicDumpConfig
     *             - Output: Not used
     *             [IoctlCommand_GetHeapStats]
     *             - Input : Not used
     *             - Output: HeapStats
     *             [IoctlCommand_StartHeapTrack]
     *             - Input : HeapTrackPoint
     *             - Output: Not used
     *             [IoctlCommand_GetHeapStats]
     *             - Input : HeapTrackPoint
     *             - Output: TrackStats
     */
    static nn::Result Ioctl(
        Output* pOutBufferInfo,
        Input* pInBufferInfo,
        IoctlCommand cmd) NN_NOEXCEPT;
};

}}
