﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/detail/service/srepo_Common.h>
#include <nn/ncm/ncm_ProgramId.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

/*!
    @brief  何らかの型を表す列挙型です。

    @see SomeValue
*/
enum class SomeValueTag : int8_t
{
    Int32,              //!< int32_t 型のタグ
    Bool,               //!< bool 型のタグ
    ProgramId,          //!< nn::ncm::ProgramId 型のタグ
    String16,           //!< String16 型のタグ
    AccountStatus,      //!< AccountStatus 型のタグ
    ControllerStatus,   //!< ControllerStatus 型のタグ
    DeviceOperationMode, //!< 本体動作モードのタグ
    SystemPowerState,   //!< SystemPowerState 型のタグ
    FriendPresence,     //!< FriendPresence 型のタグ
    CompletedNetworkRequestType, //!< CompletedNetworkRequestType 型のタグ
};

/*!
    @brief  16バイト以内の文字列を持つ型です。
*/
struct String16
{
    char value[16];

    static String16 Create(const char* p) NN_NOEXCEPT
    {
        String16 ret;
        int len = nn::util::Strlcpy(ret.value, p, sizeof(ret.value));
        NN_SDK_ASSERT_LESS(len, static_cast<int>(sizeof(ret.value)));
        NN_UNUSED(len);

        return ret;
    }

    friend bool operator==(const String16& lhs, const String16& rhs) NN_NOEXCEPT
    {
        return nn::util::Strncmp(lhs.value, rhs.value, sizeof(lhs.value)) == 0;
    }
    friend bool operator!=(const String16& lhs, const String16& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};
#if !defined( NN_BUILD_CONFIG_COMPILER_GCC )
NN_STATIC_ASSERT(std::is_trivially_copyable<String16>::value);
#endif

/*!
    @brief  アカウントのステータスを扱う型です。
*/
struct AccountStatus
{
    bool isOpened; //!< オープン状態かどうか

    static AccountStatus Create(bool isOpened) NN_NOEXCEPT
    {
        AccountStatus ret = {isOpened};
        return ret;
    }
    friend bool operator==(const AccountStatus& lhs, const AccountStatus& rhs) NN_NOEXCEPT
    {
        return lhs.isOpened == rhs.isOpened;
    }
    friend bool operator!=(const AccountStatus& lhs, const AccountStatus& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/*!
    @brief  コントローラのステータスを扱う型です。
*/
struct ControllerStatus
{
    int8_t totalNum; //!< コントローラの総数
    int8_t railNum; //!< レール接続のコントローラ数

    static ControllerStatus Create(int8_t total, int8_t rail) NN_NOEXCEPT
    {
        ControllerStatus ret = {total, rail};
        return ret;
    }
    friend bool operator==(const ControllerStatus& lhs, const ControllerStatus& rhs) NN_NOEXCEPT
    {
        return true
            && lhs.totalNum == rhs.totalNum
            && lhs.railNum == rhs.railNum
            ;
    }
    friend bool operator!=(const ControllerStatus& lhs, const ControllerStatus& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/*!
    @brief  何らかの型を保持するクラスのベースクラスです。
*/
class SomeValueBase
{
public:
    explicit SomeValueBase(SomeValueTag tag) NN_NOEXCEPT
        : m_Tag(tag)
    {
    }

    SomeValueTag GetTag() const NN_NOEXCEPT
    {
        return m_Tag;
    }

private:
    // 継承先クラスにキャストされてもコンストラクト時の値を保持するためメンバに持つ
    SomeValueTag m_Tag;
};

/*!
    @brief  何らかの型を保持するクラスです。
*/
template <typename T>
class SomeValue : public SomeValueBase
{
public:
    using ValueType = typename std::remove_cv<T>::type;

    static_assert(std::is_reference<T>::value == false, "T must not be a reference.");
    static_assert(std::is_array<T>::value == false, "T must not be a array.");
#if !defined( NN_BUILD_CONFIG_COMPILER_GCC )
    static_assert(std::is_trivially_copyable<typename std::remove_reference<ValueType>::type >::value == true, "T must be trivially copyable.");
#endif

    static const SomeValueTag Tag; //!< 保持する型を表す SomeValueTag

    /*!
        @brief  コンストラクタ
    */
    SomeValue() NN_NOEXCEPT
        : SomeValueBase(Tag)
    {
    }

    /*!
        @brief  コンストラクタ
    */
    explicit SomeValue(const T& value) NN_NOEXCEPT
        : SomeValueBase(Tag)
        , m_Value(value)
    {
    }

    /*!
        @brief  値への参照を取得します。
    */
    const T& GetRef() const NN_NOEXCEPT
    {
        return m_Value;
    }

    /*!
        @brief  値への参照を取得します。
    */
    void Set(const T& value) NN_NOEXCEPT
    {
        m_Value = value;
    }

private:
    ValueType m_Value;
};

/*!
    @brief  SomeValueオブジェクトを作成します。

    @param[in] value 作成するオブジェクトの値
*/
template <typename T>
auto CreateSomeValue(const T& value) NN_NOEXCEPT
{
    return SomeValue<T>(value);
}

/*!
    @brief  SomeValueオブジェクトの中身を表す文字列を保持するクラス
*/
class SomeValueString
{
public:
    /*!
        @brief  コンストラクタ
     */
    SomeValueString() NN_NOEXCEPT = default;

    /*!
        @brief  コンストラクタ(文字列構築付き)
     */
    explicit SomeValueString(const SomeValueBase* pSrc) NN_NOEXCEPT;

    /*!
        @brief  文字列を構築します
     */
    const char* Set(const SomeValueBase* pSrc) NN_NOEXCEPT;

    /*!
        @brief  文字列の先頭ポインタを取得します
     */
    const char* Get() const NN_NOEXCEPT
    {
        return m_pString;
    }

    friend bool operator == (const SomeValueString& lhs, const SomeValueString& rhs) NN_NOEXCEPT
    {
        if(lhs.m_pString == nullptr && rhs.m_pString == nullptr)
        {
            return true;
        }
        else if(lhs.m_pString == nullptr || rhs.m_pString == nullptr)
        {
            return false;
        }

        return nn::util::Strncmp(lhs.m_pString, rhs.m_pString, sizeof(m_Buffer)) == 0;
    }
    friend bool operator != (const SomeValueString& lhs, const SomeValueString& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

private:
    const char* m_pString = nullptr;
    char m_Buffer[32];

    const char* ToString(const SomeValueBase* pSrc, char* pBuffer, size_t size) const NN_NOEXCEPT;
};

}}}}}
