﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace srepo { namespace detail { namespace msgpack {

/*!
    @brief      MessagePack のデータ型識別子です。
*/
enum DataType : Bit8
{
    DataType_Nil            = 0xC0, //!< Nil。
    DataType_False          = 0xC2, //!< False。
    DataType_True           = 0xC3, //!< True。
    DataType_Binary8        = 0xC4, //!< バイナリ列（データサイズ 8 ビット長）。
    DataType_Binary16       = 0xC5, //!< バイナリ列（データサイズ 16 ビット長）。
    DataType_Binary32       = 0xC6, //!< バイナリ列（データサイズ 32 ビット長）。
    DataType_Extension8     = 0xC7, //!< 拡張データ列（データサイズ 8 ビット長）。
    DataType_Extension16    = 0xC8, //!< 拡張データ列（データサイズ 16 ビット長）。
    DataType_Extension32    = 0xC9, //!< 拡張データ列（データサイズ 32 ビット長）。
    DataType_Float32        = 0xCA, //!< 単精度浮動小数点数。
    DataType_Float64        = 0xCB, //!< 倍精度浮動小数点数。
    DataType_U8             = 0xCC, //!< 符号なし 8 ビット整数。
    DataType_U16            = 0xCD, //!< 符号なし 16 ビット整数。
    DataType_U32            = 0xCE, //!< 符号なし 32 ビット整数。
    DataType_U64            = 0xCF, //!< 符号なし 64 ビット整数。
    DataType_S8             = 0xD0, //!< 符号付き 8 ビット整数。
    DataType_S16            = 0xD1, //!< 符号付き 16 ビット整数。
    DataType_S32            = 0xD2, //!< 符号付き 32 ビット整数。
    DataType_S64            = 0xD3, //!< 符号付き 64 ビット整数。
    DataType_FixExtension1  = 0xD4, //!< 固定長拡張データ列（データサイズ 1 バイト）。
    DataType_FixExtension2  = 0xD5, //!< 固定長拡張データ列（データサイズ 2 バイト）。
    DataType_FixExtension4  = 0xD6, //!< 固定長拡張データ列（データサイズ 4 バイト）。
    DataType_FixExtension8  = 0xD7, //!< 固定長拡張データ列（データサイズ 8 バイト）。
    DataType_FixExtension16 = 0xD8, //!< 固定長拡張データ列（データサイズ 16 バイト）。
    DataType_String8        = 0xD9, //!< 文字列（データサイズ 8 ビット長）。
    DataType_String16       = 0xDA, //!< 文字列（データサイズ 16 ビット長）。
    DataType_String32       = 0xDB, //!< 文字列（データサイズ 32 ビット長）。
    DataType_Array16        = 0xDC, //!< 配列（要素数 16 ビット長）。
    DataType_Array32        = 0xDD, //!< 配列（要素数 32 ビット長）。
    DataType_Map16          = 0xDE, //!< 連想配列（要素数 16 ビット長）。
    DataType_Map32          = 0xDF  //!< 連想配列（要素数 16 ビット長）。
};

/*!
    @brief      MessagePack のデータ型識別子です。
*/
enum AnyDataType
{
    AnyDataType_Nil       = 0, //!< Nil。
    AnyDataType_Boolean   = 1, //!< ブール型。
    AnyDataType_Binary    = 2, //!< バイナリ列。
    AnyDataType_Extension = 3, //!< 拡張データ列。
    AnyDataType_Float32   = 4, //!< 単精度浮動小数点数。
    AnyDataType_Float64   = 5, //!< 倍精度浮動小数点数。
    AnyDataType_UInteger  = 6, //!< 符号なし整数。
    AnyDataType_SInteger  = 7, //!< 符号付き整数。
    AnyDataType_String    = 8, //!< 文字列。
    AnyDataType_Array     = 9, //!< 配列。
    AnyDataType_Map       = 10 //!< 連想配列。
};

/*!
    @brief      文字列データ情報を扱う構造体です。
*/
struct String
{
    size_t length;   //!< 文字列長。
    size_t position; //!< ストリーム内の先頭位置。
};

/*!
    @brief      バイナリデータ情報を扱う構造体です。
*/
struct Binary
{
    size_t length;   //!< データサイズ。
    size_t position; //!< ストリーム内の先頭位置。
};

/*!
    @brief      拡張データ情報を扱う構造体です。
*/
struct Extension
{
    nn::Bit8 type;   //!< 拡張データ型。
    size_t length;   //!< データサイズ。
    size_t position; //!< ストリーム内の先頭位置。
};

/*!
    @brief      任意のデータを含む構造体です。
*/
struct AnyData
{
    AnyDataType type; //!< データ型識別子。
    union
    {
        bool boolean;        //!< 真偽値。
        float f32;           //!< 単精度浮動小数点数。
        double f64;          //!< 倍精度浮動小数点数。
        int64_t si;          //!< 符号付き整数。
        uint64_t ui;         //!< 符号なし整数。
        uint32_t num;        //!< 配列、または、連想配列の要素数。
        String string;       //!< 文字列。
        Binary binary;       //!< バイナリデータ。
        Extension extension; //!< 拡張データ。
    };
};

/*!
    @brief      入力ストリーム型です。
*/
struct InputStreamParam
{
    const Bit8* data; //!< データの先頭ポインタ。
    size_t size;      //!< データサイズ。
    size_t position;  //!< 読み込み位置。

    /*!
        @brief      データの残りサイズを取得します。

        @return     データの残りサイズ。
    */
    inline size_t GetRemainSize() const NN_NOEXCEPT
    {
        return size - position;
    }
};

/*!
    @brief      出力ストリーム型です。
*/
struct OutputStreamParam
{
    Bit8* buffer;    //!< バッファの先頭ポインタ。
    size_t size;     //!< バッファサイズ。
    size_t position; //!< 書き込み位置。

    /*!
        @brief      バッファの残りサイズを取得します。

        @return     バッファの残りサイズ。
    */
    inline size_t GetRemainSize() const NN_NOEXCEPT
    {
        return size - position;
    }
};

}}}}
