﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/spsm/spsm_PowerStateTypes.h>

namespace nn { namespace spsm {

    /**
     * @brief ライブラリを初期化します。
     *
     * @post
     *  ライブラリは初期化済の状態になります。
     *
     * @details
     *  ライブラリを初期化します。 @n
     *  本関数の呼び出し回数はライブラリ内部でカウントされます。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief ライブラリを終了します。
     *
     * @post
     *  ライブラリは初期化前の状態になります。
     *
     * @details
     *  ライブラリを終了します。 @n
     *  ライブラリ初期化を行った回数分、本関数が呼ばれた場合にのみ、
     *  ライブラリが初期化前の状態になります。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   現在のシステム電源状態を取得します。
     *
     * @return  現在のシステム電源状態を返します。
     *
     * @details
     *  現在のシステム電源状態を取得します。
     */
    PowerState GetCurrentState() NN_NOEXCEPT;

    /**
     * @brief   システムをスリープ状態に移行させます。
     *
     * @details
     *  システムをスリープ状態に移行させます。
     *  この関数は、システムがスリープから復帰して、以下のシステム電源状態のいずれかに確定したときに返ります。
     *
     *   - @ref nn::spsm::PowerState_FullAwake
     *   - @ref nn::spsm::PowerState_MinimumAwake
     *   - @ref nn::spsm::PowerState_MinimumAwakeForLowBatterySign
     */
    Result SleepSystemAndWaitAwake() NN_NOEXCEPT;

    /**
     * @brief   システムの前回のスリープからの復帰時の復帰要因を取得します。
     *
     * @details
     *  システムの前回のスリープからの復帰時の復帰要因を取得します。
     */
    WakeReasonFlagSet GetLastWakeReason() NN_NOEXCEPT;

    /**
     * @brief   システムをシャットダウンします。
     *
     * @details
     *  システムをシャットダウンします。この関数は返ってきません。
     */
    NN_NORETURN void ShutdownSystem() NN_NOEXCEPT;

    /**
     * @brief   システムを再起動します。
     *
     * @details
     *  システムを再起動します。この関数は返ってきません。
     */
    NN_NORETURN void RebootSystem() NN_NOEXCEPT;

    /**
     * @brief   通知メッセージ受信可能を知らせる os::SystemEventType オブジェクトを初期化します。
     *
     * @param[in]   event   os::SystemEventType オブジェクトへのポインタ
     *
     * @pre
     *  - event が初期化されていない状態である
     *
     * @details
     *  spsm からの通知メッセージが受信可能な場合にシグナル化される
     *  nn::os::SystemEventType オブジェクトとして event を初期化します。
     *
     *  event には未初期化の nn::os::SystemEventType オブジェクトを渡して下さい。
     *  この event は、spsm から通知メッセージを受け取るとシグナル化されます。
     *  @ref nn::os::WaitSystemEvent() や @ref nn::os::WaitAny() などを使ってこれらのシグナル化を待ち受け、
     *  @ref GetNotificationMessage() を使ってメッセージを受け取り、
     *  メッセージの内容に沿った処理を行なって下さい。
     *
     *  なお、event は nn::os::EventClearMode_ManualClear で初期化されます。
     *  全ての通知メッセージを受け取ると event は自動的にクリアされるため、これを明示的にクリアする必要はありません。
     *
     */
    void InitializeNotificationMessageEvent(os::SystemEventType* event) NN_NOEXCEPT;

    /**
     * @brief   通知メッセージを取得します。
     *
     * @return  通知メッセージを示す Message 型の列挙子が返ります。
     *
     * @details
     *  通知メッセージを取得します。
     *
     *  @ref InitializeNotificationMessageEvent() で渡した SystemEventType オブジェクトがシグナル化されると、
     *  spsm から何かしらの通知メッセージが送られて来ています。本関数はその通知メッセージを取得します。
     *
     *  全ての通知メッセージを取得し終わると、 @ref InitializeNotificationMessageEvent() 時に渡した
     *  SystemEventType オブジェクトのシグナルは自動的にクリアされます。
     *
     *  通知メッセージが届いていない場合、
     *  本関数は nn::spsm::NotificationMessage_None を返します。
     */
    NotificationMessage GetNotificationMessage() NN_NOEXCEPT;

    /**
     * @brief Home ボタン長押しの時間を変更します。
     *
     * @param[in] Home ボタン長押しの時間を指定します。
     */
    void ChangeHomeButtonLongPressingTime(TimeSpan timeSpan) NN_NOEXCEPT;

    /**
     * @brief   内部ステートマシンをエラーステートに遷移させます。
     *
     * @details
     *  内部ステートマシンをエラーステートに遷移させます。
     *
     *  この関数を呼び出すと、すべてのステート遷移が中断され、以後システムを再起動しない限り一切の遷移が発生しなくなります。
     */
    void PutErrorState() NN_NOEXCEPT;
}}
