﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SPI ライブラリのバスアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>

#include "spi_Result.h"

namespace nn {
namespace spi {

/**
 * @brief セッション用パラメータ
 */
struct SpiSession
{
    void* _handle;
};

//! @name セッションの初期化と終了
//! @{

/**
 * @brief 指定したスレーブデバイスとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[out]  pOutSession                    セッション用パラメータ
 * @param[in]   deviceCode                     デバイスコード
 * @pre         ライブラリは初期化済の状態である必要があります。
 *              指定したデバイスはボード上に存在している必要があります。
 * @post        指定したデバイスが使用可能な状態になります。
 * @details     セッションのオープンに成功した場合、有効な　SpiSession　が設定されます。
 *              この関数はチップセレクト信号を使用した個々のスレーブデバイスとの接続状態の確認は行いません。
 */
void OpenSession(SpiSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT;

/**
 * @brief 指定したセッションをクローズします。
 *
 * @param[in]   session    セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがクローズ状態になります。
 * @details     オープンされていないセッションに対して実行した場合は何もしません。
 *              送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void CloseSession( const SpiSession& session ) NN_NOEXCEPT;

//! @}
// ~セッションの初期化と終了

//! @name SPI バスへのアクセス
//! @{

/**
 * @brief 指定したデータ列を送信します。
 *
 * @param[in]   session      セッション用パラメータ
 * @param[in]   pInData      送信するデータ列
 * @param[in]   dataBytes    送信するデータ列の長さ [bytes]
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ送信に成功しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @post        すべてのデータ列の送信が予約された状態になります。
 */
Result Send( const SpiSession& session, const void* pInData, size_t dataBytes ) NN_NOEXCEPT;

/**
 * @brief 指定した長さに達するまでデータ列を受信します。
 *
 * @param[out]  pOutData     受信データを格納するバッファ。
 *                           dataBytes で指定した以上のサイズで確保されている必要があります。
 * @param[in]   session      セッション用パラメータ
 * @param[in]   dataBytes    受信するデータ列の長さ [bytes]
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             指定した長さのデータ受信が完了しました。
 * @retval      ResultNoAck               対向デバイスとの接続が確立されていません。デバイスの接続状態を確認ください。
 * @retval      ResultBusBusy             バスが BUSY 状態です。通信間隔が短い場合や同一バスのトランザクションが込み合っている際に発生する可能性があります。
 * @pre         指定したデバイスとのセッションはオープン状態である必要があります。
 * @post        指定した長さのデータ列の受信が完了した状態になります。
 */
Result Receive( void* pOutData, const SpiSession& session, size_t dataBytes ) NN_NOEXCEPT;

//! @}
// ~SPI バスへのアクセス

} // spi
} // nn


