﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   オンラインストレージに関する設定 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_Types.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

const int AccountOnlineStorageSettingsCountMax =
    ::nn::account::UserCountMax;    //!< アカウント毎のオンラインストレージ設定の最大数

/**
 * @brief   アカウント毎のオンラインストレージ設定の制御フラグ定義です。
 */
struct AccountOnlineStorageFlag
{
    typedef ::nn::util::BitFlagSet<32, AccountOnlineStorageFlag>::Flag<0>
            SaveDataCheckAtLaunch; //!< 起動時に新着セーブデータを確認するか否か
};

/**
 * @brief   アカウント毎のオンラインストレージ設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AccountOnlineStorageFlag>
        AccountOnlineStorageFlagSet;

/**
 * @brief       アカウント毎のオンラインストレージ設定を表す構造体です。
 */
struct AccountOnlineStorageSettings
{
    ::nn::account::Uid uid;             //!< アカウントの識別子
    AccountOnlineStorageFlagSet flags;  //!< アカウント毎のオンラインストレージ設定の制御フラグ
    ::nn::Bit8 _reserved[4];
};

//! @name オンラインストレージ設定関連 API
//! @{

/**
 * @brief   アカウント毎のオンラインストレージ設定を取得します。
 *
 * @param[out]  outValues                   オンラインストレージ設定を読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出したオンラインストレージ設定の数を返します。
 *
 * @pre
 *              - outValues != nullptr
 *
 * @details アカウント毎のオンラインストレージ設定として指定されている値を取得可能な数だけ読み出します。
 *          取得可能なオンラインストレージ設定の数より大きなバッファ（配列）が指定された場合、
 *          余った領域に対しては何も行いません。
 *          読み出し可能なオンラインストレージ設定の最大数は AccountOnlineStorageSettingsCountMax 個です。
 */
int GetAccountOnlineStorageSettings(
    AccountOnlineStorageSettings outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       アカウント毎のオンラインストレージ設定を変更します。
 *
 * @param[in]   values                      オンラインストレージ設定（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - values != nullptr || count == 0
 *              - 0 <= count && count <= AccountOnlineStorageSettingsCountMax
 *
 * @details     オンラインストレージ設定を指定された値に変更します。
 *              オンラインストレージ設定の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能なオンラインストレージ設定の最大数は AccountOnlineStorageSettingsCountMax 個です。
 */
void SetAccountOnlineStorageSettings(
    const AccountOnlineStorageSettings values[], int count) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
