﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       EULA に関するシステムアプリケーション向け設定 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_SteadyClockTimePoint.h>

namespace nn { namespace settings { namespace system {

const int EulaVersionCountMax = 32; //!< 本体 EULA の同意バージョンの最大数

/**
 * @brief       本体 EULA の同意バージョンの時刻の種別を表す列挙体です。
 */
enum EulaVersionClockType
{
    EulaVersionClockType_NetworkSystemClock,    //!< ネットワーク時刻
    EulaVersionClockType_SteadyClock,           //!< 単調増加クロック
};

/**
 * @brief       本体 EULA の同意バージョンを表す構造体です。
 */
struct EulaVersion
{
    uint32_t version;                               //!< バージョン
    int32_t regionCode;                             //!< リージョンコード
    int32_t clockType;                              //!< 時刻の種別
    NN_PADDING4;
    ::nn::time::PosixTime networkSystemClock;       //!< 時刻 (ネットワーク時刻)
    ::nn::time::SteadyClockTimePoint steadyClock;   //!< 時刻 (単調増加クロック)
};

//! @name 本体 EULA の同意バージョン関連 API
//! @{

/**
 * @brief       本体 EULA の同意バージョンを取得します。
 *
 * @details     本体 EULA の同意バージョンとして記録されている値を取得可能な数だけ読み出します。
 *              取得可能な本体 EULA の同意バージョンの数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能な本体 EULA の同意バージョンの最大数は EulaVersionCountMax 個です。
 *
 * @param[out]  outValues                   本体 EULA の同意バージョンを読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出した本体 EULA の同意バージョンの数を返します。
 *
 * @pre
 *              - outValues != nullptr
 */
int GetEulaVersions(EulaVersion outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       本体 EULA の同意バージョンを変更します。
 *
 * @details     本体 EULA の同意バージョンを指定された値に変更します。
 *              本体 EULA の同意バージョンの変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能な本体 EULA の同意バージョンの最大数は EulaVersionCountMax 個です。
 *
 * @param[in]   values                      本体 EULA の同意バージョン（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - 0 <= count && count <= EulaVersionCountMax
 */
void SetEulaVersions(const EulaVersion values[], int count) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
