﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Bluetooth デバイス設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_Common.h>

namespace nn { namespace settings { namespace system {

const int BluetoothDevicesSettingsCountMax = 10;    //!< Bluetooth デバイス設定の最大数

const int BlePairingSettingsCountMax = 10;  //!< BLE ペアリング設定の最大数

/**
 * @brief       Bluetooth デバイス設定を表す構造体です。
 */
struct BluetoothDevicesSettings final
{
    uint8_t  bd_addr[6];           // 0-5
    char     device_name[32];      // 6-37
    uint8_t  class_of_device[3];   // 38-40
    uint8_t  link_key[16];         // 41-56
    uint8_t  link_key_present;     // 57
    uint16_t version;              // 58-59
    uint32_t trusted_services;     // 60-63
    uint16_t vid;                  // 64-65
    uint16_t pid;                  // 66-67
    uint8_t  sub_class;            // 68
    uint8_t  attribute_mask;       // 69
    uint16_t descriptor_length;    // 70-71
    uint8_t  descriptor[128];      // 72-199
    uint8_t  key_type;             // 200
    uint8_t  device_type;          // 201
    uint16_t brr_size;             // 202-203
    uint8_t  brr[9];               // 204-212
    uint8_t  reserved[299];        // 213-511 reserved for future use
};

/**
 * @brief       BLE ペアリング設定を表す構造体です。
 */
struct NN_ALIGNAS(4) BlePairingSettings final
{
    uint8_t  bd_addr[6];           // 0-5
    uint16_t manufacturer_id;      // 6-7
    uint8_t  client_id[3];         // 8-10
    uint8_t  server_id[3];         // 11-13
    uint8_t  pairing_type;         // 14
    uint8_t  reserved[113];        // 15-127 reserved for future use
};

/**
 * @brief       Bluetooth デバイス設定を取得します。
 *
 * @details     Bluetooth デバイス設定として指定されている値を取得可能な数だけ読み出します。
 *              取得可能な Bluetooth デバイス設定の数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能な Bluetooth デバイス設定の最大数は BluetoothDevicesSettingsCountMax 個です。
 *
 * @param[out]  outValues                   Bluetooth デバイス設定を読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出した Bluetooth デバイス設定の数を返します。
 *
 * @pre
 *              - outValues != nullptr
 */
int GetBluetoothDevicesSettings(
    BluetoothDevicesSettings outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       Bluetooth デバイス設定を変更します。
 *
 * @details     Bluetooth デバイス設定を指定された値に変更します。
 *              Bluetooth デバイス設定の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能な Bluetooth デバイス設定の最大数は BluetoothDevicesSettingsCountMax 個です。
 *
 * @param[in]   values                      Bluetooth デバイス設定（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - 0 <= count && count <= BluetoothDevicesSettingsCountMax
 */
void SetBluetoothDevicesSettings(
    const BluetoothDevicesSettings values[], int count) NN_NOEXCEPT;

/**
 * @brief       Bluetooth が有効か否かを表す値を返します。
 *
 * @return      Bluetooth が有効か否かを表す値です。
 */
bool IsBluetoothEnabled() NN_NOEXCEPT;

/**
 * @brief       Bluetooth が有効か否かを表す値を設定します。
 *
 * @param[in]   enable                      Bluetooth が有効か否かを表す値
 */
void SetBluetoothEnabled(bool enable) NN_NOEXCEPT;

/**
 * @brief       AFH が有効か否かを表す値を返します。
 *
 * @return      AFH が有効か否かを表す値です。
 */
bool IsBluetoothAfhEnabled() NN_NOEXCEPT;

/**
 * @brief       AFH が有効か否かを表す値を設定します。
 *
 * @param[in]   enable                      AFH が有効か否かを表す値
 */
void SetBluetoothAfhEnabled(bool enable) NN_NOEXCEPT;

/**
 * @brief       Boost が有効か否かを表す値を返します。
 *
 * @return      Boost が有効か否かを表す値です。
 */
bool IsBluetoothBoostEnabled() NN_NOEXCEPT;

/**
 * @brief       Boost が有効か否かを表す値を設定します。
 *
 * @param[in]   enable                      Boost が有効か否かを表す値
 */
void SetBluetoothBoostEnabled(bool enable) NN_NOEXCEPT;

/**
 * @brief       BLE ペアリング設定を取得します。
 *
 * @details     BLE ペアリング設定として指定されている値を取得可能な数だけ読み出します。
 *              取得可能な BLE ペアリング設定の数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能な BLE ペアリング設定の最大数は BlePairingSettingsCountMax 個です。
 *
 * @param[out]  outValues                   BLE ペアリング設定を読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出した BLE ペアリング設定の数を返します。
 *
 * @pre
 *              - outValues != nullptr
 */
int GetBlePairingSettings(
    BlePairingSettings outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       BLE ペアリング設定を変更します。
 *
 * @details     BLE ペアリング設定を指定された値に変更します。
 *              BLE ペアリング設定の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能な BLE ペアリング設定の最大数は BlePairingSettingsCountMax 個です。
 *
 * @param[in]   values                      BLE ペアリング設定（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - 0 <= count && count <= BlePairingSettingsCountMax
 */
void SetBlePairingSettings(
    const BlePairingSettings values[], int count) NN_NOEXCEPT;

}}} // nn::settings::system
