﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       画面の明るさに関するシステムアプリケーション向け設定 API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace settings { namespace system {

NN_STATIC_ASSERT(sizeof(float) == 4);

/**
 * @brief       画面の明るさの制御フラグ定義です。
 */
struct BacklightFlag
{
    typedef ::nn::util::BitFlagSet<32, BacklightFlag>::Flag<0> Automatic;   //!< 自動輝度調整の有効化
};

/**
 * @brief       画面の明るさの制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, BacklightFlag> BacklightFlagSet;

/**
 * @brief       画面の明るさの設定を表す構造体です。
 */
struct BacklightSettings
{
    BacklightFlagSet flags;                     //!< 制御フラグ
    float brightness;                           //!< 輝度設定値
    ::nn::util::Float3 brightnessCoefficients;  //!< 輝度レベルからバックライト出力への変換係数 (二次関数)
    ::nn::util::Float3 sensorCoefficients;      //!< 明るさセンサー値から輝度レベルへの変換係数 (二次関数)
    float delayDarkToBright;                    //!< 暗い状態から明るい状態へ自動輝度調整の輝度値変化への遅延
    float delayBrightToDark;                    //!< 明るい状態から暗い状態へ自動輝度調整の輝度値変化への遅延
};

/**
 * @brief       画面の明るさの設定 (VR モード対応版) を表す構造体です。
 */
struct BacklightSettingsEx
{
    BacklightFlagSet flags;                     //!< 制御フラグ
    float brightness;                           //!< 輝度設定値
    float brightnessVr;                         //!< 輝度設定値 (VR モード用)
    ::nn::util::Float3 brightnessCoefficients;  //!< 輝度レベルからバックライト出力への変換係数 (二次関数)
    ::nn::util::Float3 sensorCoefficients;      //!< 明るさセンサー値から輝度レベルへの変換係数 (二次関数)
    float delayDarkToBright;                    //!< 暗い状態から明るい状態へ自動輝度調整の輝度値変化への遅延
    float delayBrightToDark;                    //!< 明るい状態から暗い状態へ自動輝度調整の輝度値変化への遅延
};

//! @name 画面の明るさ関連 API
//! @{

/**
 * @brief       画面の明るさの設定を取得します。
 *
 * @details     画面の明るさの設定として指定されている値を取得します。
 *
 * @param[out]  pOutValue                   画面の明るさの設定を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetBacklightSettings(BacklightSettings* pOutValue) NN_NOEXCEPT;

/**
 * @brief       画面の明るさの設定を変更します。
 *
 * @details     画面の明るさの設定を指定された値に変更します。
 *
 * @param[in]   value                       画面の明るさの設定
 */
void SetBacklightSettings(const BacklightSettings& value) NN_NOEXCEPT;

/**
 * @brief       画面の明るさの設定 (VR モード対応版) を取得します。
 *
 * @details     画面の明るさの設定 (VR モード対応版) として指定されている値を取得します。
 *
 * @param[out]  pOutValue                   画面の明るさの設定 (VR モード対応版) を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetBacklightSettings(BacklightSettingsEx* pOutValue) NN_NOEXCEPT;

/**
 * @brief       画面の明るさの設定 (VR モード対応版) を変更します。
 *
 * @details     画面の明るさの設定 (VR モード対応版) を指定された値に変更します。
 *
 * @param[in]   value                       画面の明るさの設定 (VR モード対応版)
 */
void SetBacklightSettings(const BacklightSettingsEx& value) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
