﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ファームウェアデバッグ設定に関する Getter 向け API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/settings/settings_Result.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>

namespace nn { namespace settings { namespace fwdbg {

namespace detail {

struct SettingsItemKeyIteratorImpl;

} // namaspace detail

/**
 * @brief       設定項目のキーのイテレータを表す構造体です。
 */
struct SettingsItemKeyIterator
{
    ::nn::util::TypedStorage<
        detail::SettingsItemKeyIteratorImpl, 16, sizeof(void*)>  _impl;
};

/**
 * @brief       設定項目の値のバイト数を取得します。
 *
 * @details     指定された設定からキーと対応する設定項目の値のバイト数を読み出します。
 *
 * @param[in]   name                            設定の名前
 * @param[in]   key                             設定項目のキー
 *
 * @return      設定項目の値のバイト数です。
 *
 * @pre
 *              - name != nullptr
 *              - 0 < strlen(name) && strlen(name) <= SettingsNameLengthMax
 *              - name に対応する設定が存在
 *              - key != nullptr
 *              - 0 < strlen(key) && strlen(key) <= SettingsItemKeyLengthMax
 *              - key に対応する設定項目が存在
 */
size_t GetSettingsItemValueSize(const char* name, const char* key) NN_NOEXCEPT;

/**
 * @brief       設定項目の値を取得します。
 *
 * @details     指定された設定からキーと対応する設定項目の値を読み出します。
 *              読み出す値よりも大きなバッファが指定された場合、余った領域に対しては何も行いません。
 *              読み出す値よりも小さなバッファが指定された場合、先頭から順にバッファのバイト数分を読み出します。
 *
 * @param[out]  buffer                          値の読み出し先となるバッファ
 * @param[in]   bufferSize                      値の読み出し先となるバッファのバイト数
 * @param[in]   name                            設定の名前
 * @param[in]   key                             設定項目のキー
 *
 * @return      読み出した値のバイト数です。
 *
 * @pre
 *              - buffer != nullptr
 *              - name != nullptr
 *              - 0 < strlen(name) && strlen(name) <= SettingsNameLengthMax
 *              - name に対応する設定が存在
 *              - key != nullptr
 *              - 0 < strlen(key) && strlen(key) <= SettingsItemKeyLengthMax
 *              - key に対応する設定項目が存在
 *
 * @post
 *              - 戻り値は 0 以上 bufferSize 以下
 */
size_t GetSettingsItemValue(void* buffer,
                            size_t bufferSize,
                            const char* name,
                            const char* key) NN_NOEXCEPT;

/**
 * @brief       設定項目のキーのイテレータを生成します。
 *
 * @details     指定された設定に含まれる設定項目のキーのイテレータを生成します。
 *
 * @param[in]   name                            設定の名前
 *
 * @return      設定項目のキーのイテレータです。
 *
 * @pre
 *              - name != nullptr
 *              - 0 < strlen(name) && strlen(name) <= SettingsNameLengthMax
 *              - name に対応する設定が存在
 *
 * @post
 *              - 戻り値は有効なイテレータ
 */
SettingsItemKeyIterator CreateSettingsItemKeyIterator(const char* name
                                                      ) NN_NOEXCEPT;

/**
 * @brief       設定項目のキーのイテレータを破棄します。
 *
 * @param[in]   pIterator                       設定項目のキーのイテレータ
 *
 * @pre
 *              - pIterator != nullptr
 *
 * @post
 *              - *pIterator は無効なイテレータ
 */
void DestroySettingsItemKeyIterator(SettingsItemKeyIterator* pIterator
                                    ) NN_NOEXCEPT;

/**
 * @brief       設定項目のキーのイテレータを進めます。
 *
 * @details     設定項目のキーのイテレータを次のキーへ進めます。
 *
 * @param[in]   pIterator                       設定項目のキーのイテレータを指すポインタ
 *
 * @return      処理の結果が返ります。
 * @retval      ResultSuccess                   成功しました。
 * @retval      ResultStopIteration             イテレーションが終端に達しました。
 *
 * @pre
 *              - pIterator != nullptr
 *              - *pIterator は有効なイテレータ
 *
 * @post
 *              - 成功以外ならば *pIterator は無効なイテレータ
 */
::nn::Result AdvanceSettingsItemKeyIterator(SettingsItemKeyIterator* pIterator
                                            ) NN_NOEXCEPT;

/**
 * @brief       設定項目のキーのバイト数を取得します。
 *
 * @details     イテレータが示す設定項目のキーのバイト数を読み出します。
 *              バイト数には終端文字のバイト数も含まれます。
 *
 * @param[in]   iterator                        設定項目のキーのイテレータ
 *
 * @return      設定項目のキーのバイト数です。
 *
 * @pre
 *              - iterator は有効なイテレータ
 *
 * @post
 *              - 戻り値は 0 より大きく SettingsItemKeyLengthMax + 1 以下
 */
size_t GetSettingsItemKeySize(const SettingsItemKeyIterator& iterator
                              ) NN_NOEXCEPT;

/**
 * @brief       設定項目のキーを取得します。
 *
 * @details     イテレータが示す設定項目のキーを読み出します。読み出されるキーには終端文字も含まれます。
 *              読み出す値よりも大きなバッファが指定された場合、余った領域に対しては何も行いません。
 *              読み出す値よりも小さなバッファが指定された場合、先頭から順にバッファのバイト数分を読み出します。
 *              この際、終端文字が改めて付加されることはありません。
 *
 * @param[out]  buffer                          値の読み出し先となるバッファ
 * @param[in]   bufferSize                      値の読み出し先となるバッファのバイト数
 * @param[in]   iterator                        設定項目のキーのイテレータ
 *
 * @return      読み出したキーのバイト数です。
 *
 * @pre
 *              - buffer != nullptr
 *              - iterator は有効なイテレータ
 *
 * @post
 *              - 戻り値は 0 以上 ::std::min(bufferSize, SettingsItemKeyLengthMax + 1) 以下
 */
size_t GetSettingsItemKey(void* buffer,
                          size_t bufferSize,
                          const SettingsItemKeyIterator& iterator) NN_NOEXCEPT;

}}} // namespace nn::settings::fwdbg
