﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   無線 LAN の生産時設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

namespace nn { namespace settings { namespace factory {

/**
 * @brief       MAC アドレスを表す構造体です。
 */
struct MacAddress
{
    Bit8 octets[6]; //!< MAC アドレスを構成するオクテット（降順）
};

/**
 * @brief       国名コードを表す構造体です。
 */
struct CountryCode
{
    char string[3]; //!< ISO 3166-1 alpha-2 形式の国名コード（NULL 終端）
};

/**
 * @brief       無線 LAN に対して設定された MAC アドレスを取得します。
 *
 * @details     生産時に無線 LAN に対して設定された MAC アドレスを取得します。
 *
 * @param[out]  pOutValue                       MAC アドレスを読み出すバッファ
 *
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetWirelessLanMacAddress(MacAddress* pOutValue) NN_NOEXCEPT;

/**
 * @brief       無線 LAN に対して設定された国名コードの数を取得します。
 *
 * @details     生産時に無線 LAN に対して設定された国名コードの数を取得します。
 *
 * @param[out]  pOutCount                       国名コードの数を読み出すバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutCount != nullptr
 */
::nn::Result GetWirelessLanCountryCodeCount(int* pOutCount) NN_NOEXCEPT;

/**
 * @brief       無線 LAN に対して設定された国名コードを取得します。
 *
 * @details     生産時に無線 LAN に対して設定された国名コードを取得します。
 *
 * @param[out]  pOutCount                       国名コードの数を読み出すバッファ
 * @param[out]  pOutCodes                       国名コードを読み出すバッファ（配列）
 * @param[in]   count                           pOutCodes の数
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutCount != nullptr
 *              - pOutCodes != nullptr
 *              - count >= 0
 */
::nn::Result GetWirelessLanCountryCodes(int* pOutCount,
                                        CountryCode* pOutCodes,
                                        int count) NN_NOEXCEPT;

}}} // nn::settings::factory
