﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   モーションセンサの生産時設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/factory/settings_Result.h>

namespace nn { namespace settings { namespace factory {

/**
 * @brief       加速度センサのオフセット値を表す構造体です。
 */
struct AccelerometerOffset
{
    int16_t x;  //!< 加速度センサの x 軸のオフセット値
    int16_t y;  //!< 加速度センサの y 軸のオフセット値
    int16_t z;  //!< 加速度センサの z 軸のオフセット値
};

/**
 * @brief       加速度センサのスケール値を表す構造体です。
 */
struct AccelerometerScale
{
    int16_t x;  //!< 加速度センサの x 軸のスケール値
    int16_t y;  //!< 加速度センサの y 軸のスケール値
    int16_t z;  //!< 加速度センサの z 軸のスケール値
};

/**
 * @brief       ジャイロスコープのオフセット値を表す構造体です。
 */
struct GyroscopeOffset
{
    int16_t x;  //!< ジャイロスコープの x 軸のオフセット値
    int16_t y;  //!< ジャイロスコープの y 軸のオフセット値
    int16_t z;  //!< ジャイロスコープの z 軸のオフセット値
};

/**
 * @brief       ジャイロスコープのスケール値を表す構造体です。
 */
struct GyroscopeScale
{
    int16_t x;  //!< ジャイロスコープの x 軸のスケール値
    int16_t y;  //!< ジャイロスコープの y 軸のスケール値
    int16_t z;  //!< ジャイロスコープの z 軸のスケール値
};

/**
 * @brief       加速度センサのオフセット値を取得します。
 *
 * @details     生産時に設定された加速度センサのオフセット値を取得します。
 *
 * @param[out]  pOutValue                   加速度センサのオフセット値を読み出すバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAccelerometerOffset(AccelerometerOffset* pOutValue) NN_NOEXCEPT;

/**
 * @brief       加速度センサのスケール値を取得します。
 *
 * @details     生産時に設定された加速度センサのスケール値を取得します。
 *
 * @param[out]  pOutValue                   加速度センサのスケール値を読み出すバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAccelerometerScale(AccelerometerScale* pOutValue) NN_NOEXCEPT;

/**
 * @brief       ジャイロスコープのオフセット値を取得します。
 *
 * @details     生産時に設定されたジャイロスコープのオフセット値を取得します。
 *
 * @param[out]  pOutValue                   ジャイロスコープのオフセット値を読み出すバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetGyroscopeOffset(GyroscopeOffset* pOutValue) NN_NOEXCEPT;

/**
 * @brief       ジャイロスコープのスケール値を取得します。
 *
 * @details     生産時に設定されたジャイロスコープのスケール値を取得します。
 *
 * @param[out]  pOutValue                   ジャイロスコープのスケール値を読み出すバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetGyroscopeScale(GyroscopeScale* pOutValue) NN_NOEXCEPT;

}}} // nn::settings::factory
