﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace sasbus { namespace detail {

//!< リング LIFO 上の要素をスレッドセーフに扱うためのクラスです。
template<typename LifoT>
class AtomicStorage final
{
    NN_DISALLOW_COPY(AtomicStorage);
    NN_DISALLOW_MOVE(AtomicStorage);

private:
    ::std::atomic<int64_t> m_SamplingNumber;
    LifoT m_State;

public:
    AtomicStorage() NN_NOEXCEPT
        : m_SamplingNumber(0)
        , m_State()
    {
        // 何もしない
    }

    ~AtomicStorage() NN_NOEXCEPT { /* 何もしない */ }

    //!< 要素をスレッドセーフに取得します。
    bool Get(LifoT* outValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        int64_t samplingNumber = m_SamplingNumber;
        *outValue = m_State;

        // 読み出し中に値が書き換わっていなければ成功
        return (samplingNumber == m_SamplingNumber);
    }

    //!< 要素をスレッドセーフに設定します。
    void Set(const LifoT& value) NN_NOEXCEPT
    {
        m_SamplingNumber = value.samplingNumber;
        m_State = value;
    }
};

//!< リング LIFO を扱うためのクラスです。
template<typename LifoT,
         int LifoEntryCount,
         typename AtomicStorageT = AtomicStorage<LifoT>>
class RingLifo
{
    NN_STATIC_ASSERT(LifoEntryCount >= 1);
    NN_DISALLOW_COPY(RingLifo);
    NN_DISALLOW_MOVE(RingLifo);

protected:

    //!< リングバッファ上の末尾位置
    ::std::atomic<int64_t> m_Tail;

    //!< リングバッファ上の要素数
    ::std::atomic<int64_t> m_Count;

    //!< リングバッファ
    AtomicStorageT m_Buffer[LifoEntryCount + 1];

public:
    RingLifo() NN_NOEXCEPT
        : m_Tail(0)
        , m_Count(0)
    {
        // 何もしない
    }

    virtual ~RingLifo() NN_NOEXCEPT { /* 何もしない */ }

public:
    //!< リング LIFO 上の要素の型
    typedef LifoT ValueType;

public:
    //!< LIFO を空にします。
    void Clear() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    //!< LIFO に要素を追加します。
    void Append(const LifoT& value) NN_NOEXCEPT
    {
        this->AppendImpl(value);
    }

    //!< LIFO が空か否か判定します。任意のタイミングでスレッドセーフです。
    bool IsEmpty() const NN_NOEXCEPT
    {
        return (m_Count == 0);
    }

    //!< LIFO から要素を読み出します。任意のタイミングでスレッドセーフです。
    int Read(LifoT* buffer, int bufferCount) const NN_NOEXCEPT
    {
        return this->ReadImpl(buffer, bufferCount);
    }

protected:
    //!< 基準点からの相対位置を絶対位置に変換します。
    static int CalculateIndex(int index, int bias, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, bufferCount);
        NN_SDK_REQUIRES_RANGE(bias, -bufferCount, bufferCount);
        return (index + bufferCount + bias) % bufferCount;
    }

    //!< Append() の内部実装
    template<typename T>
    void AppendImpl(const T& value) NN_NOEXCEPT
    {
        // 次の末尾位置を計算
        int nextIndex = RingLifo::CalculateIndex(
            static_cast<int>(m_Tail), 1, static_cast<int>(NN_ARRAY_SIZE(m_Buffer)));

        // 末尾位置に要素を書き込み
        m_Buffer[nextIndex].Set(value);

        // 末尾位置を更新
        m_Tail = nextIndex;

        // リングバッファ上の要素数を更新
        if (m_Count < LifoEntryCount)
        {
            ++m_Count;
        }
    }

    //!< Read() の内部実装
    template<typename T>
    int ReadImpl(T* buffer, int bufferCount) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        int countMax = 0;

        while (NN_STATIC_CONDITION(true))
        {
            // buffer の書き換え後に要素数が減らない様に単調増加
            countMax = ::std::min(
                bufferCount, ::std::max(countMax, static_cast<int>(m_Count)));

            // 読み出し開始時点の末尾位置を待避
            const int tail = static_cast<int>(m_Tail);

            int bias = countMax - 1;

            while (bias >= 0)
            {
                // 古い値から順に要素の読み出し
                const int index = RingLifo::CalculateIndex(
                    tail, -bias, static_cast<int>(NN_ARRAY_SIZE(m_Buffer)));

                if (!m_Buffer[index].Get(&buffer[bias]))
                {
                    // 読み出しに失敗した場合は読み出しを中断
                    break;
                }

                if (bias + 1 < countMax)
                {
                    // 2 回目以降の読み出し後に実施

                    if ((buffer[bias].samplingNumber -
                         buffer[bias + 1].samplingNumber) != 1)
                    {
                        // 前回のサンプル番号との間に連続性がなければ、
                        // 書き込みに追いつかれてしまっているので、
                        // 読み出しを中断
                        break;
                    }
                }

                --bias;
            }

            if (bias < 0)
            {
                break;
            }

            // 最後まで読み出せていなければ最初からやり直す
        }

        return countMax;
    }
};

}}} // namespace nn::sasbus::detail
