﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief psm ドライバの API 宣言。
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/psm/psm_SystemTypes.h>
#include <nn/psm/psm_SystemProcessTypes.h>
#include <nn/psm/psm_ErrorReportTypes.h>
#include <nn/psm/driver/psm_Types.h>

namespace nn { namespace psm { namespace driver {

/**
 * @brief      psm ドライバを初期化します。
 * @post
 *  - psm ドライバが初期化済の状態である
 * @details    psm ドライバを初期化します。
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief      psm ドライバの終了処理をします。
 * @post
 *  - psm ドライバが初期化前の状態である
 * @details    psm ドライバを終了します。
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief      電池残量を取得します。
 * @param[out] pOutValue 電池残量 [%]
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  電池残量を 1 % から 100 % までの 1 % 刻みで取得します。
 */
::nn::Result GetBatteryChargePercentage(int* pOutValue) NN_NOEXCEPT;

/**
 * @brief      未加工の電池残量を取得します。
 * @param[out] pOutValue 未加工の電池残量 [%]
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  未加工の電池残量を 0 % ～ 100 % までの値で取得します。
 */
::nn::Result GetRawBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT;

/**
 * @brief      電池劣化量を取得します。
 * @param[out] pOutValue 電池劣化量 [%]
 * @pre
 *  - PSM ドライバが初期化済の状態である
 * @details
 *  電池劣化量を取得します。
 */
::nn::Result GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT;

/**
 * @brief      本体に接続されている充電器の種別を取得します。
 * @param[out] pOutValue 本体に接続されている充電器の種別
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  本体に接続されている充電器の種別を取得します。
 *  本関数は本体に充電器が接続されていないときも呼び出すことができ、
 *  その場合には ChargerType_Unconnected が取得できます。
 */
::nn::Result GetChargerType(ChargerType* pOutValue) NN_NOEXCEPT;

/**
 * @brief      電池電圧の状態を取得します。
 * @param[out] pOutValue 電池電圧の状態。
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  電池電圧の状態を取得します。
 */
::nn::Result GetBatteryVoltageState(BatteryVoltageState* pOutValue) NN_NOEXCEPT;

/**
 * @brief      充電を有効化します。
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  充電を有効化し、本体に充電器が接続されると自動的に充電を開始する状態にします。
 */
::nn::Result EnableBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      充電を無効化します。
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  充電を無効化し、本体に充電器が接続されても充電を開始しない状態にします。
 */
::nn::Result DisableBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      充電が有効化されているかどうかを取得します。
 * @param[out] pOutValue 充電が有効化されているかどうか
 * @pre
 *  - psm ドライバが初期化済の状態である
 * @details
 *  充電が有効化されているかどうかを取得します。
 */
::nn::Result IsBatteryChargingEnabled(bool* pOutValue) NN_NOEXCEPT;

/**
 * @brief      コントローラへの給電の使用権を取得します。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details
 *  コントローラへの給電の使用権を取得します。
 */
::nn::Result AcquireControllerPowerSupply() NN_NOEXCEPT;

/**
 * @brief      コントローラへの給電の使用権を解放します。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details
 *  コントローラへの給電の使用権を解放します。
 */
::nn::Result ReleaseControllerPowerSupply() NN_NOEXCEPT;

/**
 * @brief      セッションをオープンします。
 * @param[out] pOutSession セッション
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @post
 *  - 指定したセッションがオープン状態である
 * @details
 *  セッションをオープンします。
 */
::nn::Result OpenSession(Session* pOutSession) NN_NOEXCEPT;

/**
 * @brief      セッションをクローズします。
 * @param[in]  session セッション
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @post
 *  - 指定したセッションがクローズ状態である
 * @details
 *  セッションをクローズします。
 */
::nn::Result CloseSession(const Session& session) NN_NOEXCEPT;

/**
 * @brief      本体に接続されている充電器の変化とイベントを紐付けます。
 * @param[in]  pEvent  本体に接続されている充電器が変化したときシグナル化するイベント
 * @param[in]  session セッション
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details
 *  本体に接続されている充電器の変化とイベントを紐付けます。
 */
::nn::Result BindStateChangeEvent(
    ::nn::os::SystemEventType* pEvent,
    const Session& session) NN_NOEXCEPT;

/**
 * @brief      本体に接続されている充電器の変化とイベントの紐付けを解除します。
 * @param[in]  session セッション
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details
 *  本体に接続されている充電器の変化とイベントの紐付けを解除します。
 */
::nn::Result UnbindStateChangeEvent(const Session& session) NN_NOEXCEPT;

/**
 * @brief      充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  session  セッション
 * @param[in]  isEnabled 充電器の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
::nn::Result SetChargerTypeChangeEventEnabled(const Session& session, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  session  セッション
 * @param[in]  isEnabled 給電能力の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
::nn::Result SetPowerSupplyChangeEventEnabled(const Session& session, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  session  セッション
 * @param[in]  isEnabled 電池電圧の状態の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
::nn::Result SetBatteryVoltageStateChangeEventEnabled(const Session& session, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      給電エミュレーションを有効にします。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details
 *  給電エミュレーションを有効にします。
 *  給電エミュレーションが有効な期間は給電能力が十分な充電器が接続されているように振舞います。
 */
::nn::Result EnableEnoughPowerChargeEmulation() NN_NOEXCEPT;

/**
 * @brief      給電エミュレーションを無効にします。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details
 *  給電エミュレーションを無効にします。
 */
::nn::Result DisableEnoughPowerChargeEmulation() NN_NOEXCEPT;

/**
 * @brief      高速充電を有効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    高速充電を有効にします。
 */
::nn::Result EnableFastBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      高速充電を無効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    高速充電を無効にします。
 */
::nn::Result DisableFastBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      十分な給電がされているかどうかを取得します。
 * @param[out] pOutValue 十分な給電がされているかどうか
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    十分な給電がされているかどうかを取得します。
 */
::nn::Result IsEnoughPowerSupplied(bool* pOutValue) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の PmModule へのポインタを取得します。
 * @param[out] pOutPmModulePointer PmModule へのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の PmModule へのポインタを取得します。
 */
::nn::Result GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の PSC ハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer PSC ハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の PSC ハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の電池残量 IC ハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer 電池残量 IC ハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の電池残量 IC ハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の電池残量計定期的データ受信のハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer 電池残量計定期的データ受信のハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の電池残量計定期的データ受信のハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の USB PD ハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer USB PD ハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の USB PD ハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の USB PM ハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer USB PM ハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の USB PM ハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      テスト用途でドライバ内部の給電エミュレーションハンドリング完了イベントへのポインタを取得します。
 * @param[out] pOutEventPointer 給電エミュレーションハンドリング完了イベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    テスト用途でドライバ内部の給電エミュレーションハンドリング完了イベントへのポインタを取得します。
 */
::nn::Result GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

/**
 * @brief      エラーレポートのコンテキストの更新を伝えるシステムイベントへのポインタを取得します。
 * @param[out] pOutSystemEventPointer エラーレポートのコンテキスト更新を伝えるシステムイベントのポインタへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    エラーレポートのコンテキストの更新を伝えるシステムイベントへのポインタを取得します。
 */
::nn::Result GetBatteryChargeInfoEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT;

/**
 * @brief      エラーレポートのコンテキストを取得します。
 * @param[out] pOutFields エラーレポートのコンテキストへのポインタ。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    エラーレポートのコンテキストを取得します。
 */
::nn::Result GetBatteryChargeInfoFields(BatteryChargeInfoFields* pOutFields) NN_NOEXCEPT;

/**
 * @brief      電池の残量ずれを検出して補正を行ったことを伝えるシステムイベントを取得します。
 * @param[out] pOutSystemEventPointer 電池の残量ずれを検出して補正を行ったことを伝えるシステムイベントへのポインタ。
 * @param[in]  eventClearMode システムイベントのクリア方法。
 * @pre
 *  - PSM ドライバが初期化済みの状態である
 * @details    電池の残量ずれを検出して補正を行ったことを伝えるシステムイベントを取得します。
 */
::nn::Result GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT;

}}} // namespace nn::psm::driver
