﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 電池残量の生値と表示値との変換 API（boot プロセスからも参照される）。
 */

#pragma once

#include <algorithm>
#include <cmath>

#include <nn/nn_Common.h>

#include "../../psm_SystemProcessTypes.h"

namespace nn { namespace psm { namespace driver { namespace detail {

const double MinRawDefaultPercentage = 3.0;
const double MaxRawDefaultPercentage = 99.0;

// 生値 11 % を補正上限とします（デフォルトの生値の範囲で表示値 9% 相当）
const double MinRawThresholdPercentage = 11.0;

const int MinDisplayPercentage = 1;
const int MaxDisplayPercentage = 100;

// 生値の上限と下限をマージンありの上限と下限に変換します。
inline void ConvertToMarginatedRawPercentage(double* pOutMarginatedMaxRawPercentage, double* pOutMarginatedMinRawPercentage, double maxRawPercentage, double minRawPercentage) NN_NOEXCEPT
{
    // 生値最小値を 3%(Default) 以上 11%(補正上限) 以下にクランプします
    minRawPercentage = ::std::max(::std::min(minRawPercentage, MinRawThresholdPercentage), MinRawDefaultPercentage);

    // マージンの割合は旧算出式 <displayPercentage> = 1.04 * <rawPercentage> - 2.09 と同等になるようにします。
    const double MinMarginPercentage = 0.93359375; // 3.93359375 - 3.00000000(旧算出式の閾値 - MinRawDefaultPercentage)
    const double MaxMarginPercentage = -0.83593750; // 98.16406250 - 99.00000000(旧算出式の閾値 - MaxRawDefaultPercentage)

    auto marginFactor = (maxRawPercentage - minRawPercentage) / (MaxRawDefaultPercentage - MinRawDefaultPercentage);
    *pOutMarginatedMinRawPercentage = minRawPercentage + MinMarginPercentage * marginFactor;
    *pOutMarginatedMaxRawPercentage = maxRawPercentage + MaxMarginPercentage * marginFactor;
}

// 生値の上限、下限と現在値から表示値を取得します。
inline int GetDisplayPercentage(double rawPercentage, double maxRawPercentage, double minRawPercentage) NN_NOEXCEPT
{
    // 表示値は 1% ~ 100% で変化し 2.00 % 未満の時 1%、100.00 % 以上の時に 100 % となります。
    // 生値と表示値で一次式の関係を持つ区間は 2.00 % ~ 100.00 % となりますので計算の為定数 BaseDisplayPercentage を定義します。
    const double BaseDisplayPercentage = 2.00;

    auto displayPercentage = (static_cast<double>(MaxDisplayPercentage - BaseDisplayPercentage)
        * (rawPercentage - minRawPercentage)) / (maxRawPercentage - minRawPercentage) + static_cast<double>(BaseDisplayPercentage);

    // 1 以上 100 以下にクランプします。
    return ::std::max(::std::min(static_cast<int>(displayPercentage), MaxDisplayPercentage), MinDisplayPercentage);
}

//! 指定された生値と生値の範囲を元に表示値を MinDisplayPercentage から MaxDisplayPercentage の間で返します
inline int ConvertBatteryChargePercentage(double rawPercentage, double maxRawPercentage, double minRawPercentage) NN_NOEXCEPT
{
    // 生値から表示値に変換します
    double marginatedMinRawPercentage = 0.0;
    double marginatedMaxRawPercentage = 0.0;

    ConvertToMarginatedRawPercentage(&marginatedMaxRawPercentage, &marginatedMinRawPercentage, maxRawPercentage, minRawPercentage);

    return GetDisplayPercentage(rawPercentage, marginatedMaxRawPercentage, marginatedMinRawPercentage);
}

inline int ConvertBatteryChargePercentage(double rawPercentage) NN_NOEXCEPT
{
    return ConvertBatteryChargePercentage(rawPercentage, MaxRawDefaultPercentage, MinRawDefaultPercentage);
}

}}}} // namespace nn::psm::driver::detail
