﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @file
    @brief  Type definitions used by PSC controller.
*/

#pragma once

#include <nn/psc/psc_Types.h>

namespace nn      {
namespace psc     {

/**
    @brief      Generates PmFlagSet value with all flag set to false
*/
NN_FORCEINLINE PmFlagSet MakeNoPmFlags()
{
    return ::nn::util::MakeBitFlagSet<32, PmFlag>();
}

/**
    @brief      State of PSC controller, obtained with GetState()
*/
enum PmControlState
{
    /**
        @brief      PSC is in idle state and may accept new request.
    */
    PmControlState_Idle,

    /**
        @brief      PSC has finished task but encountered an error.
        @details    When error condtion is detected, PSC will change its
                    internal state to ControlState_Error and will signal
                    an event to PM control service. Error type can be
                    retrieved using GetResult()
    */
    PmControlState_Error,

    /**
        @brief      PSC is running a task.
        @details    PSC will use this state when it's in the middle
                    of dispatching request to managed modules.
                    Once task is completed, state will change to
                    either idle or error and an event will be signaled
                    to PM control service.
    */
    PmControlState_Running,
};

/**
    @brief      Controls order in which requests are dispatched to PmModules.
    @details    The order in which requests are dispatched to PmModules
                depends on whether transition is from a lower to
                higher power consumption state or not. In order to transition
                into a lower state, all higher level modules in the dependency
                tree should transition first. For transitions into higher power
                consumption states, lower level modules should transition before
                higher level modules.
*/
enum PmTransitionOrder
{
    /**
        @brief  Dispatch requests in ascending order, from the top most to bottom PmModule in dependency chain.
    */
    PmTransitionOrder_ToLowerPowerState,

    /**
        @brief  Dispatch requests in descneding order, from the bottom PmModule to the top most in dependency chain.
    */
    PmTransitionOrder_ToHigherPowerState
};

}}
