﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief Nintendo NX CPU Profiler General functions in the API
 * @details Nintendo NX CPU Profiler API
 */

#pragma once

#include <nn/nn_Common.h>

#include <nn/ro/ro_Types.h>

namespace nn { namespace profiler {

//! @name Relocatable Module Tracking API
//! @{


/**
 *  @brief Informs the profiler that a relocatable module has been loaded.
 *
 *  @param [in] pModule A pointer to the module that was loaded.
 *
 *  @details
 *  The Nintendo NX CPU Profiler has a limited ability to track relocatable objects
 *  that are loaded or unloaded while profiling.
 *  However, if a module is both loaded and unloaded during profiling, the profiler
 *  will not be able to properly track functions from that object.
 *  This function provides the user with a mechanism to inform the profiler that
 *  a relocatable object was loaded.
 *  This also allows for better handling of the relocatable object inside the PC tool,
 *  as the load and unload times are able to be factored in.
 *
 *  This function must be called after the call to nn::ro::LoadModule().
 *
 *  Example:
 *  @code
 *  nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
 *  nn::profiler::RecordLoadModule(&module);
 *  @endcode
 */
void RecordLoadModule(nn::ro::Module* pModule) NN_NOEXCEPT;


/**
 *  @brief Informs the profiler that a relocatable module has been unloaded.
 *
 *  @param [in] pModule A pointer to the module that was unloaded.
 *
 *  @details
 *  The Nintendo NX CPU Profiler has a limited ability to track relocatable objects
 *  that are loaded or unloaded while profiling.
 *  However, if a module is both loaded and unloaded during profiling, the profiler
 *  will not be able to properly track functions from that object.
 *  This function provides the user with a mechanism to inform the profiler that
 *  a relocatable object was unloaded.
 *  This also allows for better handling of the relocatable object inside the PC tool,
 *  as the load and unload times are able to be factored in.
 *
 *  This function must be called after the call to nn::ro::UnloadModule().
 *
 *  Example:
 *  @code
 *  nn::ro::UnloadModule(&module);
 *  nn::profiler::RecordUnloadModule(&module);
 *  @endcode
 */
void RecordUnloadModule(nn::ro::Module* pModule) NN_NOEXCEPT;

//! @}

}} // nn::profiler
