﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/util/util_Span.h>
#include <nn/prepo/detail/service/core/prepo_Storage.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      ファイルシステムライブラリのラッパーモジュールです。
*/
class FileSystem
{
public:
    /*!
        @brief      FS の管理ブロックのサイズです。
    */
    static const int64_t FsMetaDataBlockSize = 16 * 1024;

    /*!
        @brief      エントリー情報格納領域のサイズです。
    */
    static const int64_t EntryInfoSize = FsMetaDataBlockSize * 2;

private:
    static nn::util::Span<Storage>& GetStorages() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(Storage, s_Array, [] =
        {
            // マウント名   セーブデータ ID     合計サイズ          ジャーナルサイズ
            { "prepo-sys",  0x80000000000000A0,  1 * 1024 * 1024,   128 * 1024 }, // システム情報管理ストレージ
            { "prepo",      0x80000000000000A1, 10 * 1024 * 1024,   512 * 1024 }, // 通常のレポートのデータ管理ストレージ
            { "prepo-ap",   0x80000000000000A2,  1 * 1024 * 1024,   256 * 1024 }  // 海賊版対策のレポートのデータ管理ストレージ
        });
        NN_FUNCTION_LOCAL_STATIC(auto, s_Span, = nn::util::MakeSpan(s_Array));
        return s_Span;
    }

public:
    /*!
        @brief      テストモードを有効にします。

        @details
                    テストモードが有効な場合、ファイルシステムに Host FS を利用します。

                    テストモードを有効にするときは、ファイルシステムを利用する前に、この関数を呼び出してください。

                    Host FS を利用するには、desc の以下の項目に Debug を追加する必要があります。
                    - Desc/FsAccessControlDescriptor/FlagPresets
                    - Default/FsAccessControlData/FlagPresets
    */
    static void EnableTestMode() NN_NOEXCEPT;

    /*!
        @brief      ストレージをマウントします。

        @param[in]  mountName   マウント名。

        @return     処理結果。
    */
    static nn::Result Mount(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンマウントします。

        @param[in]  mountName   マウント名。
    */
    static void Unmount(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをロックします。

        @param[in]  mountName   マウント名。
    */
    static void Lock(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンロックします。

        @param[in]  mountName   マウント名。
    */
    static void Unlock(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの使用量を取得します。

        @param[in]  mountName   マウント名。

        @return     使用量。
    */
    static int64_t GetStorageUsage(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @param[out] outSize     空き容量。
        @param[in]  mountName   マウント名。

        @return     処理結果。

        @pre
            - outSize != nullptr
    */
    static nn::Result GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの全体容量を取得します。

        @param[in]  mountName   マウント名。

        @return     全体容量。
    */
    static int64_t GetTotalSpaceSize(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージが自スレッドによってロックされているか否かを返します。

        @param[in]  mountName   マウント名。

        @return     処理結果。
    */
    static bool IsLockedByCurrentThread(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      システムが利用するストレージをすべてマウントします。

        @details
                    1 つでもマウントに失敗した場合、ABORT します。
    */
    static void MountAll() NN_NOEXCEPT;

    /*!
        @brief      システムが利用するストレージをすべてアンマウントします。
    */
    static void UnmountAll() NN_NOEXCEPT;

    /*!
        @brief      ストレージをコミットします。

        @param[in]  mountName   マウント名。

        @return     処理結果。

        @details
                    ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    static nn::Result Commit(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをロールバックします。

        @param[in]  mountName   マウント名。

        @return     処理結果。

        @details
                    ロールバック中にストレージにアクセスしてはいけません。
    */
    static nn::Result Rollback(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に作成します。

        @param[in]  path        パス。
        @param[in]  isFilePath  ファイルパスを指定したかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    ファイルパスを指定した場合、親ディレクトリの生成を行います。
    */
    static nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath = false) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に削除します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    static nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;

    /*!
        @brief      ストレージの内容をすべて削除します。

        @param[in]  mountName   マウント名。

        @pre
            - mountName != nullptr

        @details
                    ストレージが空のときは、何もせずに成功します。
    */
    static void ClearStorage(const char* mountName) NN_NOEXCEPT;

private:
    static Storage& FindStorage(const char* mountName) NN_NOEXCEPT;
};

/*!
    @brief      ストレージを局所的にロックします。
*/
#define NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK(mountName) \
    detail::service::core::FileSystem::Lock(mountName);         \
    NN_UTIL_SCOPE_EXIT                                          \
    {                                                           \
        detail::service::core::FileSystem::Unlock(mountName);   \
    }

/*!
    @brief      ストレージのロックをアサートします。
*/
#define NN_DETAIL_PREPO_ASSERT_STORAGE_LOCK(mountName) \
    NN_SDK_ASSERT(detail::service::core::FileSystem::IsLockedByCurrentThread(mountName))

}}}}}
