﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  PlayDataManager から情報を取得する API を定義します。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/util/util_Optional.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_QueryApi.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/pdm/pdm_Types.h>

namespace nn { namespace pdm {

/**
* @brief        アプリケーションの起動・終了に関するイベントを取得します。
* @param[out]   outValue 出力されたイベント
* @param[in]    outMaxCount 出力できるイベントの最大数（outValue のサイズ）
* @param[in]    eventIndexOffset 取得を開始するイベント全体の先頭からのオフセット。取得対象に含まれます。
* @return       出力されたイベントの数
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
*               - eventIndexOffset >= 0
* @details      システムが記録しているイベント全体を eventIndexOffset 番目から走査し、アプリケーションの起動・終了に関するイベントを出力します。
*               システムはアプリケーションの起動・終了以外のイベントも記録しており、eventIndexOffset はそれらを含めた全てのイベントの列の中での値です。
*               出力のイベントのインデックスも同様に、全てのイベントを通してのインデックス値であるため、連続的な値にはなりません。
*/
int QueryApplicationEvent(ApplicationEvent outValue[], int outCountMax, int eventIndexOffset) NN_NOEXCEPT;

/**
* @brief        アカウントのオープン・クローズのイベントを取得します。
* @param[out]   outValue 出力されたイベント
* @param[in]    outMaxCount 出力できるイベントの最大数（outValue のサイズ）
* @param[in]    eventIndexOffset 取得を開始するイベント全体の先頭からのオフセット。取得対象に含まれます。
* @return       出力されたイベントの数
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
*               - eventIndexOffset >= 0
* @details      システムが記録しているイベント全体を eventIndexOffset 番目から走査し、アカウントのオープン・クローズのイベントを出力します。
*               システムはアカウントのオープン・クローズ以外のイベントも記録しており、eventIndexOffset はそれらを含めた全てのイベントの列の中での値です。
*               出力のイベントのインデックスも同様に、全てのイベントを通してのインデックス値であるため、連続的な値にはなりません。
*/
int QueryAccountEvent(AccountEvent outValue[], int outCountMax, int eventIndexOffset) NN_NOEXCEPT;

/**
* @brief        システム上で起動されたアプリケーションの統計情報を取得します。
* @param[out]   outValue 出力された統計情報
* @param[in]    outCountMax 出力できる統計情報の最大数（outValue のサイズ）
* @return       出力された統計情報の数
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
* @details      起動されたアプリケーションの数が outCountMax を超えている場合、以下のルールで出力に残す統計情報を決定します。
*               - TBD.
*/
int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax) NN_NOEXCEPT;

/**
* @brief        指定したユーザーアカウントをオープンしたアプリケーションの統計情報を取得します。
* @param[out]   outValue 出力された統計情報
* @param[in]    outCountMax 出力できる統計情報の最大数（outValue のサイズ）
* @param[in]    user 統計情報を取得するユーザーアカウント
* @return       出力された統計情報の数
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
* @details      起動されたアプリケーションの数が outCountMax を超えている場合、以下のルールで出力に残す統計情報を決定します。
*               - TBD.
*/
int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax, const account::Uid& user) NN_NOEXCEPT;

/**
* @brief        （未実装）指定したネットワークサービスアカウントが紐付いたユーザーアカウントをオープンしたアプリケーションの統計情報を取得します。
* @param[out]   outValue 出力された統計情報
* @param[in]    outCountMax 出力できる統計情報の最大数（outValue のサイズ）
* @param[in]    networkServiceAccountid 統計情報を取得するネットワークサービスアカウント
* @return       出力された統計情報の数
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
* @details      起動されたアプリケーションの数が outCountMax を超えている場合、以下のルールで出力に残す統計情報を決定します。
*               - TBD.
*/
int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax, const account::NetworkServiceAccountId& networkServiceAccountId) NN_NOEXCEPT;

/**
* @brief        指定したアプリケーションの統計情報を取得します。
* @param[in]    applicationId 統計情報を取得するアプリケーションのID
* @return       指定したアプリケーションの統計情報
* @details      指定したアプリケーションを起動した記録がない場合、無効状態の値が返ります。
*/
nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId) NN_NOEXCEPT;

/**
* @brief        指定したユーザーアカウント、指定したアプリケーションの統計情報を取得します。
* @param[in]    applicationId 統計情報を取得するアプリケーションのID
* @param[in]    user 統計情報を取得するユーザーアカウント
* @return       指定したユーザーアカウント、指定したアプリケーションの統計情報
* @details      指定したアプリケーションで指定したユーザーアカウントが開かれた記録がない場合、無効状態の値が返ります。
*/
nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId, const account::Uid& user) NN_NOEXCEPT;

/**
* @brief        （未実装）指定したネットワークサービスアカウント、指定したアプリケーションの統計情報を取得します。
* @param[in]    applicationId 統計情報を取得するアプリケーションのID
* @param[in]    networkServiceAccountid 統計情報を取得するユーザーアカウント
* @return       指定したネットワークサービスアカウント、指定したアプリケーションの統計情報
* @details      指定したアプリケーションで指定したネットワークサービスアカウントが紐付いた状態のユーザーアカウントが開かれた記録がない場合、無効状態の値が返ります。
*/
nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId, const account::NetworkServiceAccountId& networkServiceAccountId) NN_NOEXCEPT;

/**
    @brief      指定したアプリケーションの統計情報を取得します。
    @param[out] outValues 最近遊んだアプリケーションのアプリケーションID
    @param[in]  applicationIds out に格納できるアプリケーションIDの数
    @param[in]  count outValues, applicationIds の要素数
    @return     outValues に格納したアプリケーションの数
*/
int QueryApplicationPlayStatisticsForSystem(ApplicationPlayStatistics outValues[], const ncm::ApplicationId applicationIds[], int count) NN_NOEXCEPT;

/**
    @brief      指定したユーザーが最近遊んだアプリケーションを取得します。
    @param[out] out 最近遊んだアプリケーションのアプリケーションID
    @param[in]  outCount out に格納できるアプリケーションIDの数
    @param[in]  user ユーザー
    @return     out に格納したアプリケーションの数
    @details    out[0] に最も最近遊んだアプリケーションが格納され、インデックスが大きいほど最後に遊んだ機会が過去のアプリケーションになります。
*/
int QueryRecentlyPlayedApplication(ncm::ApplicationId out[], int outCount, const account::Uid& user) NN_NOEXCEPT;

/**
    @brief      いずれかのユーザーの最近遊んだアプリケーションが更新されたときにシグナルされるシステムイベントを取得します。
    @return     いずれかのユーザーの最近遊んだアプリケーションが更新されたときにシグナルされるシステムイベントのポインタ
    @details    イベントのシグナルのタイミングは、正確には「いずれかのユーザーに対してアプリケーション関連のイベントが登録された時」です。
                そのため、ユーザーが同じアプリケーションを連続で遊んだ場合など、イベントのシグナル前後で @ref QueryRecentlyPlayedApplication で取得できる値が変わらない可能性があります。@n
                このシステムイベントのクリアモードは nn::os::EventClearMode_Manual に設定されます。
                シグナルは自動的にはクリアされません。更新を待ち受ける状態に入る前に明示的にクリアしてください。@n
                現在は nd サービス専用の機能です。
*/
os::SystemEvent* GetRecentlyPlayedApplicationUpdateEvent() NN_NOEXCEPT;

}}
