﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  PlayDataManager から情報を取得するデバッグ用の API を定義します。
 */

#include <nn/nn_Common.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_SystemTypes.h>

namespace nn { namespace pdm {

/**
* @brief        記録されたプレイイベントを取得します。
* @param[out]   outValue 出力。
* @param[in]    outMaxCount 出力できるイベントの最大数（outValue のサイズ）。
* @param[in]    offset 取得を開始するイベントの先頭からのオフセット。
* @return       出力されたイベントの数。
* @pre
*               - outCountMax > 0
*               - outValue に outCountMax 分以上の配列が確保されている
*               - offset >= 0
*/
int QueryPlayEvent(PlayEvent outValue[], int outCountMax, int offset) NN_NOEXCEPT;

/**
* @brief        記録されたプレイイベントの数を取得します。
* @param[out]   outStartIndex 取得可能なイベントの先頭のインデックス。
* @param[out]   outLastIndex 取得可能なイベントの末尾のインデックス。
* @return       イベントの数。
*/
int GetAvailablePlayEventRange(int* outStartIndex, int* outLastIndex) NN_NOEXCEPT;

/**
 * @brief       記録されたアカウントプレイイベントを取得します。
 *
 * @param[out]  outValue    取得先バッファ。
 * @param[in]   outCountMax 取得先バッファ格納可能イベント上限数。( outValue の容量を @ref AccountPlayEvent 単位で示します )
 * @param[in]   offset      取得対象イベント開始位置オフセット。( @ref AccountPlayEvent 単位 )
 * @param[in]   user        取得対象ユーザーアカウントID。
 *
 * @return      outValue に取得した有効なイベント数。
 *
 * @pre
 *              - outValue != nullptr
 *              - outValue に outCountMax 分以上の配列が確保されている。
 *              - outCountMax > 0
 *              - offset >= 0
 *              - user != account::InvalidUid
 */
int QueryAccountPlayEvent(AccountPlayEvent outValue[], int outCountMax, int offset, const account::Uid& user) NN_NOEXCEPT;

/**
 * @brief       記録されたアカウントプレイイベントの数を取得します。
 *
 * @param[in]   outStartIndex   データベースに保存されている取得可能イベントの先頭インデックス。
 * @param[in]   outLastIndex    データベースに保存されている取得可能イベントの末尾インデックス。
 *
 * @return      イベント数。
 *
 * @pre
 *              - outStartIndex != nullptr
 *              - outLastIndex != nullptr
 *              - user != account::InvalidUid
 */
int GetAvailableAccountPlayEventRange(int* outStartIndex, int* outLastIndex, const account::Uid& user) NN_NOEXCEPT;

}}
