﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  PlayDataManager にイベントの発生を通知するための API を定義します。
 */

#include <nn/account/account_Types.h>
#include <nn/applet/applet_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/nn_Result.h>
#include <nn/oe/oe_OperationModeApis.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/pdm/pdm_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pdm {

/**
* @brief        Notify 系の API を利用するための初期化を行います。
* @details      Notify 系の API を呼び出す前に少なくとも1度呼び出す必要があります。2度目以降の呼び出しでは初期化処理は行われません。
*               複数回呼び出した場合、Notify 系の API の利用を完全に終了するには同じ回数だけ @ref FinalizeForNotification を呼ぶ必要があります。
*/
void InitializeForNotification() NN_NOEXCEPT;

/**
* @brief        Notify 系の API の利用を終了します。
* @details      Notify 系の API の利用を完全に終了するには @ref InitializeForNotification を呼んだ回数と同じ回数だけこの関数を呼びだす必要があります。
*/
void FinalizeForNotification() NN_NOEXCEPT;

/**
* @brief        アプレットに関するイベントの発生を PlayDataManager に通知します。
* @param[in]    eventType イベントの種類。
* @param[in]    programId イベント対象のアプレットのプログラムID。
* @param[in]    version イベント対象のアプレットのバージョン。
* @param[in]    appletId イベント対象のアプレットのアプレットID。
* @param[in]    storageId イベント対象のアプレットが格納されたストレージ。
* @param[in]    logPolicy イベント対象のアプレットのログ記録方針。
*
* @details      AppletManager 向けの機能です。
*               アプレットの状態を遷移させた際に呼び出してください。
*               ライブラリアプレットについては NotifyLibraryAppletEvent() を使用してください。
*/
void NotifyAppletEvent(AppletEventType eventType, nn::ncm::ProgramId programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT;

/**
* @brief        ライブラリアプレットに関するイベントの発生を PlayDataManager に通知します。
* @param[in]    eventType イベントの種類。
* @param[in]    mainProgramId イベント対象のアプレットの呼び出し元のアプレットのプログラムID。
* @param[in]    libraryAppletMode イベント対象のアプレットのモード。
* @param[in]    appletId イベント対象のアプレットのアプレットID。
* @param[in]    storageId イベント対象のアプレットが格納されたストレージ。
* @param[in]    logPolicy イベント対象のアプレットのログ記録方針。
*
* @details      AppletManager 向けの機能です。
*               ライブラリアプレットの状態を遷移させた際に呼び出してください。
*/
void NotifyLibraryAppletEvent(AppletEventType eventType, nn::ncm::ProgramId mainProgramId, applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT;

/**
* @brief        動作モードの変更に関するイベントの発生を PlayDataManager に通知します。
* @param[in]    operationMode 変更後の動作モード。
*
* @details      AppletManager 向けの機能です。
*				システムの電源がオンになった際、及び、動作モードの変更が発生した際に、現在の（変更後の）動作モードを引数に与えて呼び出してください。
*/
void NotifyOperationModeChangeEvent(oe::OperationMode operationMode) NN_NOEXCEPT;

/**
* @brief        電源状態の変更に関するイベントの発生を PlayDataManager に通知します。
* @param[in]    eventType イベントの種類。
*
* @details      AppletManager 向けの機能です。
*				電源状態の変更が発生した際に、現在の（変更後の）電源状態を引数に与えて呼び出してください。
*/
void NotifyPowerStateChangeEvent(PowerStateChangeEventType eventType) NN_NOEXCEPT;

/**
* @brief        全てのイベントを削除します（デバッグ用）
*/
void NotifyClearAllEvent() NN_NOEXCEPT;

/**
 * @brief       指定ユーザーアカウントのユーザーアカウントイベント記録サービスを停止します。
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess, 処理に成功しました。}
 *      @handleresult{nn::pdm::ResultServiceNotAvailable, 指定ユーザーアカウントのユーザーアカウントイベント記録サービスが見つかりませんでした。}
 * @endresult
 *
 * @details     データベース永続用記録媒体であるシステムセーブデータのマウントを解除します。
 *              本APIは、アカウントセーブデータの常時マウントによる弊害を回避するためのものであり、停止中は以下の制限があります。
 *              出来るだけ速やかに再開してください。
 *                  - ユーザーアカウント( @ref nn::account::Uid ) に依存した QueryPlayStatistics 経由のクエリに失敗します。
 *                  - ユーザーアカウント依存のイベント記録が一時保存領域を越えて発生した場合、対象のイベント記録が破棄されます。
 *              呼び出し回数はネスト管理されます。
 */
Result SuspendUserAccountEventService(const account::Uid& user) NN_NOEXCEPT;

/**
 * @brief       指定ユーザーアカウントのユーザーアカウントイベント記録サービスを再開します。
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess, 処理に成功しました。}
 *      @handleresult{nn::pdm::ResultServiceNotAvailable, 指定ユーザーアカウントのユーザーアカウントイベント記録サービスが見つかりませんでした。}
 * @endresult
 *
 * @details     データベース永続用記録媒体であるシステムセーブデータをマウントします。
 *              呼び出し回数はネスト管理されます。
 */
Result ResumeUserAccountEventService(const account::Uid& user) NN_NOEXCEPT;

/**
 * @brief       サービス停止/再開隠蔽ユーティリティ。
 */
template<typename Predicate>
Result PresentUserAccountEventServiceSuspend(const account::Uid& uid, Predicate predicate) NN_NOEXCEPT
{
    // pdm がマウントするユーザ依存システムセーブデータのアンマウントを行います。
    bool skipResume = false;
    NN_RESULT_TRY(SuspendUserAccountEventService(uid))
        NN_RESULT_CATCH(ResultServiceNotAvailable)
        {
            // サービス停止対象(ユーザ)が見つからないため再開をスキップします。
            skipResume = true;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        if (!skipResume)
        {
            ResumeUserAccountEventService(uid);
        }
    };
    return predicate();
}

}}
