﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <mutex>
#include <tuple>
#include <utility>
#include <nn/account.h>
#include <nn/os.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_UniqueLock.h>

namespace nn { namespace pdm { namespace detail {

struct InitializationManager
{
    nn::os::SdkMutexType _mutex;
    int _count;

    void lock() NN_NOEXCEPT
    {
        _mutex.lock();
    }

    void unlock() NN_NOEXCEPT
    {
        _mutex.unlock();
    }

    void Initialize(std::function<void()> initializer) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(*this);
        if( _count == 0 )
        {
            initializer();
        }
        _count++;
    }

    void Finalize(std::function<void()> finalizer) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(*this);
        if( _count > 0 )
        {
            _count--;
        }

        if( _count == 0 )
        {
            finalizer();
        }
    }
};

#define NN_PDM_INITIALIZATION_INITIALIZER { NN_OS_SDK_MUTEX_INITIALIZER(), 0 }

uint32_t Hi(uint64_t value) NN_NOEXCEPT;
uint32_t Low(uint64_t value) NN_NOEXCEPT;

bool IsApplicationEvent(const PlayEvent& playEvent) NN_NOEXCEPT;
bool IsApplicationBecomeForegroundEvent(const AppletEventType& type) NN_NOEXCEPT;
bool IsApplicationBecomeForegroundEvent(const PlayEvent& playEvent, bool statisticsMustBeAvailable) NN_NOEXCEPT;

bool IsLibraryApplet(applet::AppletId id) NN_NOEXCEPT;

nn::ncm::ProgramId GetProgramId(const PlayEvent& playEvent) NN_NOEXCEPT;
bool MatchProgramId(const nn::ncm::ProgramId& programId, const PlayEvent& playEvent) NN_NOEXCEPT;

nn::ncm::ApplicationId GetApplicationId(const PlayEvent& playEvent) NN_NOEXCEPT;
bool MatchApplicationId(const nn::ncm::ApplicationId& applicationId, const PlayEvent& playEvent) NN_NOEXCEPT;

nn::account::Uid GetUid(const PlayEvent& playEvent) NN_NOEXCEPT;
bool MatchUid(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT;
bool IsTargetUidOpenEvent(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT;
bool IsTargetUidCloseEvent(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT;

nn::account::NetworkServiceAccountId GetNetworkServiceAccountId(const PlayEvent& playEvent) NN_NOEXCEPT;
bool MatchNetworkServiceAccountId(const nn::account::NetworkServiceAccountId& networkServiceAccountId, const PlayEvent& playEvent) NN_NOEXCEPT;

bool IsPowerOffEvent(const PlayEvent& playEvent) NN_NOEXCEPT;
bool IsPowerOnEvent(const PlayEvent& playEvent) NN_NOEXCEPT;

void ConvertAppletEventToApplicationEvent(ApplicationEvent* outValue, const PlayEvent& playEvent) NN_NOEXCEPT;

/**
 * @brief   os::SdkRecursiveMutex 実装のスコープ外部排他連携用オブジェクト管理クラス。
 */
typedef util::UniqueLock<os::SdkRecursiveMutex> LockGuard;

/**
 * @brief   コピー禁止、std::tuple 管理コンテナ。
 * @details 複数の @ref util::UniqueLock 管理での利用を想定したtupleコンテナです。
 *
 * @ref     std::tuple
 */
template<class... Args>
class DisallowCopyTuple : public std::tuple<Args...>
{
    NN_DISALLOW_COPY(DisallowCopyTuple);
    typedef std::tuple<Args...> TupleType;  //!< ベース tuple 型.

public:
    /**
     * @brief   可変長パラメタコンストラクタ
     */
    explicit DisallowCopyTuple(Args&&... args) NN_NOEXCEPT
        : TupleType(std::move(args)...) {}

    /**
     * @brief   ムーブコンストラクタ
     */
    DisallowCopyTuple(DisallowCopyTuple&& rvalue) NN_NOEXCEPT
        : TupleType(std::move(rvalue)) {}

    /**
     * @brief   要素数取得
     */
    size_t GetSize() const NN_NOEXCEPT
    {
        return std::tuple_size<TupleType>::value;
    }
};

/**
 * @brief   簡易クライアント未稼働状態追跡クラス。
 */
class ClientIdleTracker
{
    NN_DISALLOW_COPY(ClientIdleTracker);
    NN_DISALLOW_MOVE(ClientIdleTracker);

public:
    typedef util::UniqueLock<ClientIdleTracker> ManagedUnlock;

    /**
     * @brief   コンストラクタ
     */
    ClientIdleTracker() NN_NOEXCEPT = default;

    /**
     * @brief   クライアントが稼働状態を獲得します。
     */
    void Lock() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        ++m_BusyCount;
    }

    /**
     * @brief   クライアントが稼働状態を獲得します。
     *
     * @details @ref Lock() と等価です。
     */
    void lock() NN_NOEXCEPT
    {
        Lock();
    }

    /**
     * @brief   クライアントが稼働状態を解除します。
     */
    void Unlock() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        const auto count = m_BusyCount - 1;
        if (0 == count)
        {
            m_BusyCount = 0;
            m_Condition.Signal();
        }
        else if (0 < count)
        {
            m_BusyCount = count;
        }
    }

    /**
     * @brief   クライアントが稼働状態を解除します。
     *
     * @details @ref Unlock() と等価です。
     */
    void unlock() NN_NOEXCEPT
    {
        Unlock();
    }

    /**
     * @brief   稼働中の全クライアントが未稼働状態になるまで待機します。
     */
    void Wait() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        while (m_BusyCount > 0)
        {
            m_Condition.Wait(m_Lock);
        }
    }

    /**
     * @brief   稼働中の全クライアントが未稼働状態になるまで時限指定で待機します。
     *
     * @return  タイムアウトした場合は、タイムアウト時の稼働中クライアント数。@n
     *          タイムアウトしなかった場合は、ゼロが返されます。
     */
    int32_t Wait(const TimeSpan& timeout) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        const auto noTimeout = os::ConditionVariableStatus_NoTimeout;
        while (m_BusyCount > 0 && noTimeout == m_Condition.TimedWait(m_Lock, timeout));
        return m_BusyCount;
    }

    /**
     * @brief   クライアントが稼働状態を獲得します。
     *
     * @return  スコープアウトで @ref unlock() を呼び出す自動解放オブジェクトを返します。
     *
     * @details @ref Lock() と同等の獲得を行います。
     */
    ManagedUnlock Acquire() NN_NOEXCEPT
    {
        return ManagedUnlock(*this);
    }

private:
    mutable os::SdkRecursiveMutex       m_Lock;         //!< 条件変数排他用ミューテックス
    mutable os::SdkConditionVariable    m_Condition;    //!< 条件変数
    volatile int32_t                    m_BusyCount = 0;//!< 稼働クライアント数カウンタ
};


}}}
