﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pdm/detail/pdm_EventEntryStream.h>
#include <nn/pdm/detail/pdm_Config.h>

namespace nn { namespace pdm { namespace detail {

/**
 * @brief    PlayEvent の書き込み・読み込みを管理するクラスです。
 */
class PlayEventBuffer : public EventEntryFileStream
{
    NN_DISALLOW_COPY(PlayEventBuffer);
    NN_DISALLOW_MOVE(PlayEventBuffer);

public:
    typedef EventEntryCache<PlayEvent>  Buffer;

    /**
     * @brief   プレイレポート用のデータ。
     */
    struct ReportData
    {
        uint32_t count;
        uint32_t startIndex;
        uint32_t lastIndex;
    };

    /**
     * @brief       インスタンスを取得します。
     */
    static PlayEventBuffer& GetInstance() NN_NOEXCEPT
    {
        static PlayEventBuffer instance;
        return instance;
    }

    /**
     * @brief       ファイルを読み込み、記録されたイベント数などの情報を初期化します。他の関数を呼ぶ前に必ず一度呼ぶ必要があります。
     * @param[in]   eventCountMax ファイルに保存できるイベントの最大数。
     * @pre
     *              - eventCountMax > 0
     * @details     ファイルに保存できるイベントの最大数の変更は、本クラスの単体テスト時の利用を意図したものです。
     */
    void Initialize(uint32_t eventCountMax = PlayEventCountMax) NN_NOEXCEPT;

    /**
     * @brief       ファイルを削除し、バッファに追加されていたイベントを破棄します。
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief       イベントをバッファに追加します。
     * @param[in]   playEvent バッファに追加するイベント。
     * @details     バッファに追加されたイベントは @ref Flush() が実行されるまでファイルには書き込まれず、@ref Read() の対象になりません。
     *              Get 系の関数で得られるインデックス値やカウントにも反映されません。@n
     *              イベント追加用のバッファに空きがない場合は、関数内部でファイルへのフラッシュ処理を行い、
     *              空き領域を確保して追加を行います。
     */
    void Add(const pdm::PlayEvent& playEvent) NN_NOEXCEPT;

    /**
     * @brief       記録されたイベントを読み込みます。事前に BeginRead を呼ぶ必要があります。
     * @param[out]  outValue 読み込み先。
     * @param[in]   startIndex 読み込みを開始するイベントのインデックス。
     * @param[in]   outCountMax 読み込むイベントの最大数。
     * @return      読み込んだイベントの数。
     * @pre
     *              - outValue != nullptr
     */
    inline uint32_t Read(pdm::PlayEvent outValue[], uint32_t startIndex, uint32_t outCountMax) const NN_NOEXCEPT
    {
        return EventEntryFileStream::ReadImpl(outValue, startIndex, outCountMax);
    }

    /**
     * @brief       イベントの追加が行われた際に Signal() を発行するイベントオブジェクトを設定します。
     * @param[in]   addedEvent イベントの追加が行われた際に Signal() を発行するイベントオブジェクト。
     */
    void SetPlayEventAddedEvent(os::Event* addedEvent) NN_NOEXCEPT;

    /**
     * @brief       バッファに追加されたイベントをファイルに書き込みフラッシュします。完了後、イベント追加用のバッファは空になります。
     */
    void Flush() NN_NOEXCEPT;

    /**
     * @brief       バッファ中にある未フラッシュのイベントの数を取得します。
     * @return      イベントの数。
     */
    uint32_t GetFilledBufferSpaceCount() const NN_NOEXCEPT;

    /**
     * @brief       バッファに溜められるイベントの最大数を取得します。
     * @return      イベントの最大数。
     */
    inline static uint32_t GetBufferSpaceCountMax() NN_NOEXCEPT
    {
        return Buffer::Capacity;
    }

    /**
     * @brief       プレイレポート用のデータを返します。
     */
    void GetReportData(ReportData* pOut) const NN_NOEXCEPT;

private:
    /**
     * @brief       コンストラクタです。
     */
    PlayEventBuffer() NN_NOEXCEPT;

    Buffer::Switcher    m_EntryCache;
    os::Event*          m_pPlayEventAddedEvent;
    bool                m_IsInitialized;
};

}}}
