﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/pctl_ApiSystemReadOnly.h>
#include <nn/pctl/pctl_TypesWatcher.h>
#include <nn/pctl/pctl_ResultWatcher.h>
#include <nn/pctl/pctl_PairingInfo.h>

#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/time/time_Types.h>

namespace nn { namespace pctl {

/**
 * @name 見守り機能サーバー連携処理関数
 * @{
 */

/**
 * @brief 連携が既に行われているかどうかを返します。
 * @return 連携済みである場合は true
 *
 * @details
 * 本関数はサーバー側の状態を確認せず、本体に連携情報があるかどうかで判定を行います。
 * サーバー側の状態については、@ref RetrievePairingInfo 関数で
 * 連携先の情報を取得することで確認を行ってください。
 *
 * 本関数はメンテナンスモードでも動作し、本体上の連携情報を参照してその結果を返します。
 */
bool IsPairingActive() NN_NOEXCEPT;

/**
 * @brief ペアコンの設定を最後に同期した時刻を返します。
 * @param[out] outTime 同期した時刻を受け取る nn::time::PosixTime のポインター
 * @return 同期済みであれば true、未連携や未同期状態であるなど返すべき値が無い場合は false
 *
 * @pre
 *  - outTime != nullptr
 *
 * @details
 * 本関数は同期した時刻を nn::time::PosixTime の形式で返します。
 * 返される値の詳細については nn::time::PosixTime の説明をご覧ください。
 *
 * 本関数はメンテナンスモードでは常に false を返します。
 */
bool GetSettingsLastUpdated(nn::time::PosixTime* outTime) NN_NOEXCEPT;

/**
 * @brief 連携先の相手情報を取得します。
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 取得処理が完了しました。
 * @retval nn::pctl::ResultPairingDeleted サーバー側で連携が解除されているため連携情報を削除しました。
 * @retval nn::pctl::ResultCanceled 通信処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @retval nn::pctl::ResultServerResourceNotAvailable 保持している連携状態の情報が利用できないため連携情報を削除しました。
 *
 * @pre
 *  - pInfo != nullptr
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - 戻り値が nn::ResultSuccess である場合
 *    - pInfo のインスタンスが有効な状態になる(@ref PairingInfo::DebugIsInstanceValid() == true)
 *
 * @details
 * 本関数は通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。@n
 * ただし、もともと連携が行われていない場合はすぐ処理を返し、
 * pInfo のインスタンスを有効な状態として nn::ResultSuccess を返します。
 * (この場合 pInfo の @ref PairingInfo::GetState は @ref PairingState_None を返します。)@n
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 本関数が nn::ResultSuccess を返した場合、pInfo から有効なデータを得ることができますが、
 * 状態が変化するとデータの取得ができなくなるため、
 * 必要なデータを取得したら pInfo を使用しないようにしてください。
 * データが取得できなくなる条件については @ref PairingInfo の説明をご覧ください。
 *
 * 関数の戻り値が nn::pctl::ResultPairingDeleted および
 * nn::pctl::ResultServerResourceNotAvailable である場合、以降
 * 連携されていない状態として扱われる(@ref IsPairingActive の戻り値が false になります)ため、
 * 連携状態を前提とした処理を行っていた場合はそれを取り止める必要があります。
 */
nn::Result RetrievePairingInfo(PairingInfo* pInfo) NN_NOEXCEPT;

/**
 * @brief 連携されている状態を解消し、サーバーに通知します。
 * @param[in] force サーバーと通信できない状況であっても連携解除を行う場合は true。
 *   この場合本関数は常に true を返します。
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 連携が解除されました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @post
 *  - 連携情報が破棄されている
 *  - 本体内の制限対象外リストが削除されている
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために事前に通信の利用要求を
 * システムに出す必要があります。(force が true であっても
 * この処理は必要ですが、利用要求が拒否された場合でも
 * そのまま本関数を呼び出すことができます。)
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 本関数は内部で保持している連携情報を破棄する前にサーバーへ通知を行います。
 * force が false の場合、通知処理に失敗するとそこでエラーとなり、連携状態は
 * 解除されないままとなります。
 * なお、本体初期化時など本体設定データがクリアされる場面では
 * 他の設定とともに連携情報が破棄されるため、本関数を呼び出す必要はありません。
 *
 * なお、未連携の状態で本関数を呼び出した場合は何もせずに true を返します。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
nn::Result UnlinkPairing(bool force) NN_NOEXCEPT;

/**
 * @brief 連携情報を破棄し、連携されていない状態にします。
 *
 * @post
 *  - 連携情報が破棄されている
 *  - 本体内の制限対象外リストが削除されている
 *
 * @details
 * 本関数は連携解除した旨をサーバーに通知しない(通信を行わない)点で
 * @ref UnlinkPairing と異なります。
 * 通信の利用要求を出さずに、通信せずに連携情報を破棄したい場面で使用してください。
 * (通信エラーを無視したいだけである場合は @ref UnlinkPairing の
 * force パラメーターを true にして該当関数を呼び出してください。)
 *
 * 本関数はメンテナンスモードでも正しく動作し、本体に保存されている
 * 連携情報を破棄する処理を行います。
 */
void DeletePairing() NN_NOEXCEPT;

/**
 * @}
 */
/**
 * @name 見守り機能同期処理関数
 * @{
 */

/**
 * @brief ペアコン設定をサーバーに同期します。
 * @param[out] outLastUpdatedTime 同期したデータの最終更新時刻(不要である場合は nullptr)
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 処理に成功しました。
 * @retval nn::pctl::ResultPairingDeleted サーバー側で連携が解除されているため連携情報を削除しました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @retval nn::pctl::ResultServerResourceNotAvailable 保持している連携状態の情報が利用できないため連携情報を削除しました。
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
nn::Result SynchronizeParentalControlSettings(nn::time::PosixTime* outLastUpdatedTime) NN_NOEXCEPT;

/**
 * @brief ペアコン設定をサーバーに同期します。
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 処理に成功しました。
 * @retval nn::pctl::ResultPairingDeleted サーバー側で連携が解除されているため連携情報を削除しました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @retval nn::pctl::ResultServerResourceNotAvailable 保持している連携状態の情報が利用できないため連携情報を削除しました。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
inline nn::Result SynchronizeParentalControlSettings() NN_NOEXCEPT
{
    return SynchronizeParentalControlSettings(nullptr);
}

/**
 * @brief 指定されたアプリケーションを制限対象外リストに追加する更新を行った後、ペアコン設定をサーバーに同期します。
 * @param[in] applicationId 制限対象外リストに追加するアプリケーションの ID
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 処理に成功しました。
 * @retval nn::pctl::ResultPairingNotActive 連携されていません。
 * @retval nn::pctl::ResultPairingDeleted サーバー側で連携が解除されているため連携情報を削除しました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @retval nn::pctl::ResultServerResourceNotAvailable 保持している連携状態の情報が利用できないため連携情報を削除しました。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
nn::Result RequestUpdateExemptionList(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @brief ペアコン設定が同期された際に通知を受け取るためのイベントオブジェクトを返します。
 * @return イベントオブジェクトインスタンスへのポインター
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - 戻り値のインスタンスは初期化された状態であり @ref nn::os::SystemEvent::Wait などで
 *    シグナルを待つことが可能である
 *
 * @details
 * サーバー側でペアコン設定が更新されて本体に反映された直後のタイミングで、
 * 本関数が返すイベントオブジェクトがシグナル状態となります。
 *
 * 本関数が返すイベントオブジェクトはシグナルを自動でクリアするモードとなります。
 * (シグナル状態のイベントオブジェクトに対して @ref nn::os::SystemEvent::Wait
 * などを呼び出すとシグナル状態がリセットされます。)
 *
 * 本関数が返すオブジェクトのインスタンスは、同一プロセスにおいては
 * 常に同一のインスタンスとなります。(異なるプロセスでは
 * 状態管理が独立した別のイベントオブジェクトを返します。)
 * グローバルに存在するインスタンスであるため、delete 演算子や
 * メモリ解放などの関数にインスタンスへのポインターを渡さないでください。
 */
nn::os::SystemEvent* GetSynchronizationEvent() NN_NOEXCEPT;

/**
 * @brief サーバーとの連携が解除されたことを得るためのイベントオブジェクトを返します。
 * @return イベントオブジェクトインスタンスへのポインター
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - 戻り値のインスタンスは初期化された状態であり @ref nn::os::SystemEvent::Wait などで
 *    シグナルを待つことが可能である
 *
 * @details
 * サーバーと連携済みである状態から通信処理の結果によって連携されていない状態に遷移したとき、
 * 本関数が返すイベントオブジェクトがシグナル状態となります。
 *
 * 本関数が返すイベントオブジェクトのシグナルは Wait 系関数などで
 * 自動ではクリアされません。シグナルは以下のいずれかに該当する場合にクリアされます。
 *
 * - ペアコン設定が無効化された(@ref nn::pctl::DeleteSettings または @ref nn::pctl::SetPinCode で nullptr が指定された)
 * - ペアコン設定が無効化される前に再度連携が行われた
 * - @ref nn::pctl::ClearUnlinkedEvent が呼び出された
 *
 * イベントオブジェクトのシグナル状態が変化するのは以上の場合のみであり、
 * それ以外の操作が行われた場合にシグナル状態が変化することはありません。@n
 * 例として、通信処理を伴わない連携解除(@ref nn::pctl::DeletePairing が呼び出された場合など)においては
 * イベントはシグナル状態になりません。
 *
 * なお、シグナルを手動でクリアする場合は @ref ClearUnlinkedEvent 関数を
 * 用いて行ってください。(直接クリアした場合、本体再起動後に再び
 * シグナルされる可能性があります。)
 *
 * 本関数が返すオブジェクトのインスタンスは、同一プロセスにおいては
 * 常に同一のインスタンスとなります。(異なるプロセスでは
 * 状態管理が独立した別のイベントオブジェクトを返します。)
 * グローバルに存在するインスタンスであるため、delete 演算子や
 * メモリ解放などの関数にインスタンスへのポインターを渡さないでください。
 */
nn::os::SystemEvent* GetUnlinkedEvent() NN_NOEXCEPT;

/**
 * @brief @ref GetUnlinkedEvent で返されるイベントのシグナル状態をクリアします。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * @ref GetUnlinkedEvent で返されるイベントをクリアする際に利用します。
 * 通常はペアコン設定の削除を行うことでシグナルをクリアすることができますが、
 * 何らかの理由でシグナルのみをクリアしたい場合はこの関数を利用してください。
 * (イベントオブジェクトに対して直接クリアした場合、本体再起動後に
 * 再びシグナルされる可能性があります。)
 */
void ClearUnlinkedEvent() NN_NOEXCEPT;

/**
 * @}
 */
/**
 * @name プレイタイマー処理関数
 * @{
 */

/**
 * @brief プレイタイマーの処理を開始します。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * プレイタイマー設定が存在しない場合は「残り時間」の計算を行いませんが、
 * 「経過時間」の計算は開始します。
 * 非同期の設定変更などによってタイマー設定が追加された場合に
 * 「残り時間」が「経過時間」を元に再計算されるため、
 * プレイタイマーを動作させたい場面では設定の有無に関わらず
 * 本関数を呼び出すようにしてください。
 */
void StartPlayTimer() NN_NOEXCEPT;

/**
 * @brief プレイタイマーの処理を停止します。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * ペアコンの一時解除状態になった際にもプレイタイマーの処理が停止します。
 * 本関数で変更される状態とは別に管理されるため、本関数を呼び出さずに
 * 一時解除状態から元の状態に戻るとプレイタイマーの処理が再開します。
 * 逆に、一時解除の状態で本関数を呼び出し、その後一時解除がリセットされると
 * プレイタイマーの処理は停止したままとなります。
 */
void StopPlayTimer() NN_NOEXCEPT;

/**
 * @brief プレイタイマーの時間切れによって制限されているかどうかを返します。
 * @return 制限されている場合は true
 *
 * @details
 * プレイタイマーのモードが @ref PlayTimerMode_Suspend の場合のみ、
 * 残り時間が 0 になった時点で利用が制限されます。@n
 * モードが PlayTimerMode_Suspend ではない場合や
 * 残り時間が 0 より大きい場合は false が返ります。
 *
 * 本体がメンテナンスモードで起動されている場合、本関数は false を返します。
 */
bool IsRestrictedByPlayTimer() NN_NOEXCEPT;

/**
 * @brief プレイタイマーの現在の設定値を取得します。
 * @param[out] pSettings 設定値を受け取るポインター
 *
 * @pre
 *  - pSettings != nullptr
 *
 * @details
 * 本体がメンテナンスモードで起動されている場合、本関数は pSettings を 0 で初期化します。
 */
void GetPlayTimerSettings(PlayTimerSettings* pSettings) NN_NOEXCEPT;

/**
 * @brief プレイタイマーの時間切れによる強制中断を行うためのイベントオブジェクトを返します。
 * @return イベントオブジェクトインスタンスへのポインター
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - 戻り値のインスタンスは初期化された状態であり @ref nn::os::SystemEvent::Wait などで
 *    シグナルを待つことが可能である
 *
 * @details
 * 本関数が返すイベントオブジェクトは、プレイタイマーのモードが
 * @ref PlayTimerMode_Suspend の場合でかつ残り時間が 0 になった
 * タイミングでシグナル状態になります。このイベントオブジェクトによる通知を
 * 受け取ることで、強制中断モードで残り時間 0 の場合に割り込み処理を行うことができます。
 *
 * 本関数が返すイベントオブジェクトはシグナルを自動でクリアするモードとなります。
 * (シグナル状態のイベントオブジェクトに対して @ref nn::os::SystemEvent::Wait
 * などを呼び出すとシグナル状態がリセットされます。)
 *
 * 本関数が返すオブジェクトのインスタンスは、同一プロセスにおいては
 * 常に同一のインスタンスとなります。(異なるプロセスでは
 * 状態管理が独立した別のイベントオブジェクトを返します。)
 * グローバルに存在するインスタンスであるため、delete 演算子や
 * メモリ解放などの関数にインスタンスへのポインターを渡さないでください。
 */
nn::os::SystemEvent* GetPlayTimerEventToRequestSuspension() NN_NOEXCEPT;

/**
 * @brief プレイタイマーのアラームが無効になっているかどうかを返します。
 * @return 無効になっている場合は true
 *
 * @details
 * 本関数が返す値が変わる場合、@ref GetSynchronizationEvent が返すイベントがシグナル状態になります。
 *
 * 本体がメンテナンスモードで起動されている場合、本関数は false を返します。
 */
bool IsPlayTimerAlarmDisabled() NN_NOEXCEPT;

/**
 * @brief (デバッグ用) プレイタイマーの現在の設定値を変更します。
 * @param[in] settings 設定値を保持するインスタンス
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 通常はサーバーとの連携によって設定が行われるため、本体システム側から
 * 設定変更を行うことはありません。デバッグ用として、サーバー連携なしに
 * プレイタイマーの設定を行いたい場合に使用してください。
 *
 * なお、settings.isEnabled の値は settings.dailySettings と
 * settings.weekSettings の値から自動的に計算されるため、直接
 * true/false の値を設定しても無視されます。
 *
 * 本機能はペアコンの制限設定が有効である場合にのみ動作し、
 * 設定が無効である場合は何も行いません。
 * また、プレイタイマー設定後に制限設定が削除された場合は
 * プレイタイマー設定も削除されます。
 */
void SetPlayTimerSettingsForDebug(const PlayTimerSettings& settings) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) プレイタイマーのアラームが無効かどうかを設定します。
 * @param[in] isDisabled 無効にする場合は true
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 通常はサーバーとの連携によって設定が行われるため、本体システム側から
 * 設定変更を行うことはありません。デバッグ用として、サーバー連携なしに
 * プレイタイマーのアラーム無効設定を行いたい場合に使用してください。
 *
 * 本機能はペアコンの制限設定が有効である場合、かつ
 * ネットワーク時計が有効である場合にのみ動作します。
 * ペアコン制限設定が無効である場合は何も行いません。
 * また、本設定後にペアコン制限設定が削除された場合は
 * 本設定も削除されます。
 *
 * なお、アラームを無効にした設定は本体の日付設定に基づく日付変更時に
 * 元の(無効ではない)設定に戻ります。
 */
void SetPlayTimerAlarmDisabledForDebug(bool isDisabled) NN_NOEXCEPT;

/**
 * @}
 */
/**
 * @name その他のサーバー連携関連関数
 * @{
 */

/**
 * @brief ペアコンサービスが処理中のネットワーク処理を中止します。
 *
 * @pre
 *  - 本体の起動モードがメンテナンスモードではない
 *
 * @details
 * 本関数は以下の関数が通信している場合にその処理を中止させます。
 *
 * - @ref RequestPairing
 * - @ref AuthorizePairing
 * - @ref RetrievePairingInfo
 * - @ref UnlinkPairing
 * - @ref SynchronizeParentalControlSettings
 * - @ref RequestUpdateExemptionList
 * - @ref PairingAccountInfo::GetMiiImage
 * - @ref PairingAccountInfo::GetMiiImageContentType
 *
 * 各関数は処理が中止されると失敗扱いとなります。
 */
void CancelNetworkRequest() NN_NOEXCEPT;

/**
 * @}
 */

}}
