﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/pctl_ResultSystem.h>

#include <nn/nn_TimeSpan.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/pctl/pctl_ApiSettings.h>
#include <nn/pctl/pctl_ApiSnsPost.h>
#include <nn/pctl/pctl_TypesSystemReadOnly.h>

namespace nn { namespace pctl {

/**
 * @name 本体機能向けペアコン状態取得関数
 * @{
 */

/**
 * @brief ペアコンの設定に基づいて汎用設定利用・変更が可能かどうかを確認します。
 * @return nn::Result の値
 * @retval nn::ResultSuccess 利用・変更が可能です(制限されていない状態です)
 * @retval nn::pctl::ResultRestricted 利用が制限されています
 *
 * @details
 * ペアコン設定画面や本体初期化画面など、ペアコンが有効な状態において
 * 制限したい場面に入る際に利用します。
 */
nn::Result ConfirmSystemSettingsPermission() NN_NOEXCEPT;

/**
 * @brief 制限が一時的に解除されているかどうかを返します。
 * @return 一時解除済みであれば true
 */
bool IsRestrictionTemporaryUnlocked() NN_NOEXCEPT;

/**
 * @brief 汎用設定利用・変更の許可が必要なシステムのシーンに入っているかどうかを返します。
 * @return 制限の必要な該当シーンに入っていれば true
 *
 * @details
 * ペアコンの制限がない場合は「制限が必要」ではないため、許可が必要なシステムのシーンに
 * 入ったかどうかにかかわらず false を返します。@n
 * 制限が行われている場合は、該当シーンに入っているかどうかで戻り値が決定されます。
 */
bool IsRestrictedSystemSettingsEntered() NN_NOEXCEPT;

/**
* @brief 現在制限がかかっている項目の種類を返します。
* @return @ref Feature の各値を組み合わせたもの。
*   制限がない場合は @ref Feature_None の値がそのまま返ります。
*/
Feature GetRestrictedFeatures() NN_NOEXCEPT;

/**
* @brief ペアコンの制限設定に基づいて指定した動画が再生可能かどうかを確認します。
* @param[in] applicationId 再生可能かどうか判定したい動画の元となったアプリケーションのアプリケーションID
* @param[in] ratingAge 再生可能かどうか判定したい動画が持つレーティング情報
* @return nn::Result の値
* @retval nn::ResultSuccess 動画の再生が可能です(制限されていないか一時解除状態です)
* @retval nn::pctl::ResultPlayingApplicationVideoRestricted レーティングによって動画再生機能が制限されています
*
* @details
* 本関数は一時解除状態であるかどうかも加味した値を返します。
* 一時解除状態である場合は nn::ResultSuccess が返ります。
*/
nn::Result ConfirmPlayableApplicationVideo(nn::ncm::ApplicationId applicationId, const int8_t (&ratingAge)[32]) NN_NOEXCEPT;

/**
 * @}
 */

/**
 * @name 本体機能向けプレイタイマー状態取得関数
 * @{
 */

/**
 * @brief プレイタイマーの処理が動いているかどうかを返します。
 * @return 処理が動いている場合は true
 *
 * @details
 * 残り時間が 0 以下になっても処理が続いている場合があります。
 *
 * 本体システム起動時(初期状態)は false の状態となります。
 *
 * 本体がメンテナンスモードで起動されている場合、本関数は false を返します。
 */
bool IsPlayTimerEnabled() NN_NOEXCEPT;

/**
 * @brief プレイタイマーの残り時間を取得します。
 * @return 残り時間
 *
 * @details
 * 戻り値の時間が負数である場合は超過していることを表します。
 * また、時間制限が存在しない場合は 0 が返ります。
 *
 * 本体がメンテナンスモードで起動されている場合、本関数は 0 を返します。
 */
nn::TimeSpan GetPlayTimerRemainingTime() NN_NOEXCEPT;

/**
 * @}
 */

}}
