﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/pctl/pctl_ApiSettings.h>
#include <nn/pctl/pctl_ResultStereoVision.h>

namespace nn { namespace pctl {

/**
 * @name 立体視機能のペアレンタルコントロール制限チェック関数
 * @{
 */

/**
 * @brief ペアレンタルコントロール制限設定に基づいて立体視機能の利用が可能かどうかを確認します。
 * @return nn::Result の値
 * @retval nn::ResultSuccess 立体視機能の利用が可能です(制限されていないか一時解除状態です)
 * @retval nn::pctl::ResultStereoVisionRestricted 立体視機能制限の設定により制限されています
 * @retval nn::pctl::ResultRestricted 上記以外の理由で機能が制限されています
 *
 * @post 立体視機能の使用が許可されている状態に遷移します。
 *
 * @details
 * 本関数は @ref SetStereoVisionRestriction で設定される
 * 制限設定を確認し、制限が必要である場合に対応する Result 値を返します。
 * アプリケーションは、アプリケーション内で立体視機能を利用しようとする
 * 場合に本関数を呼び出し、本関数が nn::ResultSuccess 以外を返した場合は
 * 機能の利用が制限されているとみなしてその機能を利用できないようにする必要があります。@n
 * なお nn::ResultSuccess 以外の戻り値である場合、その戻り値を使って
 * エラー表示関数を呼び出すことにより、具体的な制限内容を表示させることができます。
 *
 * 本関数は一時的に制限が解除されているかどうかも加味した値を返します。
 * 一時解除状態である場合は nn::ResultSuccess が返ります。
 *
 * nn::ResultSuccess が返った場合、立体視機能の使用が許可された状態になります。
 * この状態は @ref ResetConfirmedStereoVisionPermission によってリセットすることができます。
 *
 * 戻り値の @ref nn::pctl::ResultStereoVisionRestricted は
 * @ref nn::pctl::ResultRestricted をハンドルすることで自動的にハンドルすることができます。
 */
nn::Result ConfirmStereoVisionPermission() NN_NOEXCEPT;


/**
 * @brief 立体視機能の使用許可状態をリセットします。
 *
 * @post 立体視機能の使用が許可されていない状態に遷移します。
 *
 * @details
 * 本関数は @ref ConfirmStereoVisionPermission の成功によって成立した立体視機能の使用許可状態をリセットし、
 * 一度も立体視機能を使用していない状態（正確には、使用可能な状態に遷移したことがない状態）に戻します。
 *
 * リセット直後の状態、つまり @ref ConfirmStereoVisionPermission が一度も成功していないのと同等の状態では、
 * アプリケーションの中断からの再開時に、立体視機能の使用が許可されているか否かの確認が行われません。
 *
 * 例えば、一時解除を利用して立体視機能を用いるモードを開始した際に @ref ConfirmStereoVisionPermission を呼びだした後、
 * 立体視を用いないモードに遷移した場合に本関数を呼び出すことで、
 * 立体視を用いないモードでのスリープからの復帰時にペアレンタルコントロールの解除を求められる、といった体験を避けることができます。
 *
 */
void ResetConfirmedStereoVisionPermission() NN_NOEXCEPT;

/**
 * @brief ペアレンタルコントロール制限設定に基づいて立体視機能の利用が可能かどうかを確認します。
 * @return 立体視機能が利用可能か否かの値
 *
 * @details
 * 本関数は、立体視機能の利用が可能かどうかを確認します。
 * @ref ConfirmStereoVisionPermission とは異なり、立体視機能の使用が許可されている状態に遷移しません。
 * また、 @ref GetStereoVisionRestriction とは異なり、一時的に制限が解除されているかどうかも加味した値を返します。
 * 一時解除状態である場合は true が返ります。
 */
bool IsStereoVisionPermitted() NN_NOEXCEPT;


/**
 * @}
 */
/**
 * @name 立体視機能のペアレンタルコントロール制限設定変更関数
 * @{
 */

/**
 * @brief 立体視機能の制限設定(変更)が可能であるかどうかを返します。
 * @return nn::Result の値
 * @retval nn::ResultSuccess 立体視機能の制限設定が可能です
 * @retval nn::pctl::ResultRestrictionNotEnabled ペアレンタルコントロールが有効ではないため設定変更できません
 * @retval nn::pctl::ResultConfigurationNotAllowed 上記以外の理由で設定の変更が許可されていません
 *
 * @details
 * 本関数は立体視機能制限の設定変更を行うことが可能かどうかをチェックする際に用います。
 * nn::ResultSuccess が返った場合、ユーザーに設定変更画面を表示するなどして
 * 設定変更を促すことができます。@n
 * nn::ResultSuccess 以外の値が返った場合は、その戻り値を使ってエラー表示関数を呼び出すことにより、
 * 設定変更ができない詳細な理由をユーザーに案内することができます。
 *
 * なお、実際に設定変更を行う際は @ref RequestStereoVisionRestrictionConfigurationPermission を
 * 呼び出して暗証番号入力を促し、設定変更可能かどうかを確認を行う必要があります。
 *
 * 戻り値の @ref nn::pctl::ResultRestrictionNotEnabled は
 * @ref nn::pctl::ResultConfigurationNotAllowed をハンドルすることで自動的にハンドルすることができます。
 * なお、今後の本体更新によって、nn::pctl::ResultConfigurationNotAllowed に含まれるものの
 * 戻り値の説明には無い Result 値が返る可能性があります。
 */
nn::Result ConfirmStereoVisionRestrictionConfigurable() NN_NOEXCEPT;

/**
 * @brief 立体視機能の制限設定(変更)を行うための許可を要求します。
 * @return nn::Result の値
 * @retval nn::ResultSuccess 立体視機能の制限設定が可能です
 * @retval nn::pctl::ResultRestrictionNotEnabled ペアレンタルコントロールが有効ではないため設定変更できません
 * @retval nn::pctl::ResultConfigurationCanceled ユーザーによって設定変更が取り消されました
 * @retval nn::pctl::ResultConfigurationNotAllowed 上記以外の理由で設定の変更が許可されていません
 *
 * @details
 * 本関数は立体視機能の制限設定を行う直前に、設定変更が可能かどうかをチェックする際に用います。
 * 本関数は内部で暗証番号入力をするための UI 表示を行い、暗証番号が正しく入力された場合に限り
 * nn::ResultSuccess を返します。nn::ResultSuccess が返った場合、@ref SetStereoVisionRestriction による
 * 設定変更を行うことができます。@n
 * nn::ResultSuccess 以外の値が返った場合は、その戻り値を使ってエラー表示関数を呼び出すことにより、
 * 設定変更ができない詳細な理由をユーザーに案内することができます。
 * なお、暗証番号入力がキャンセルされた場合には @ref nn::pctl::ResultConfigurationCanceled が返るため、
 * この Result に対してエラー表示が不要である場合はそれに応じたハンドリング処理が必要です。
 *
 * 本関数では UI 表示を行うため、呼び出し元をブロックします。
 *
 * 戻り値のうち @ref nn::pctl::ResultRestrictionNotEnabled と @ref nn::pctl::ResultConfigurationCanceled は
 * @ref nn::pctl::ResultConfigurationNotAllowed をハンドルすることで自動的にハンドルすることができます。
 * なお、今後の本体更新によって、nn::pctl::ResultConfigurationNotAllowed に含まれるものの
 * 戻り値の説明には無い Result 値が返る可能性があります。
 */
nn::Result RequestStereoVisionRestrictionConfigurationPermission() NN_NOEXCEPT;

/**
 * @brief 立体視制限の設定値を返します。
 * @return 立体視制限の設定値。true であれば制限が有効です。
 *
 * @details
 * 本関数の戻り値は一時的に制限が解除されているかどうかは加味されず、
 * @ref SetStereoVisionRestriction で設定される純粋な設定値を表します。
 *
 * なお、一度も SetStereoVisionRestriction が呼び出されていない場合や
 * ペアレンタルコントロールの設定(暗証番号設定)がされていない場合は false が返ります。
 */
bool GetStereoVisionRestriction() NN_NOEXCEPT;

/**
 * @brief 立体視の制限を設定します。
 * @param[in] restricted 機能を制限するかどうかを示す値。制限する場合は true を指定します。
 *
 * @details
 * 設定された値はペアレンタルコントロール設定の削除によってリセットされます。
 *
 * 本関数はペアレンタルコントロールの設定(暗証番号設定)が
 * 行われている場合に呼び出すことを想定しています。
 * ペアレンタルコントロールの設定が行われていないなど、
 * @ref ConfirmStereoVisionRestrictionConfigurable() が IsSuccess() != true となるような
 * 戻り値を返す状況である場合、本関数は何も行いません。
 */
void SetStereoVisionRestriction(bool restricted) NN_NOEXCEPT;

/**
 * @}
 */

}}
