﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/npns/npns_Types.h>
#include <nn/pctl/detail/service/common/pctl_AsyncContext.h>
#include <nn/pctl/detail/service/common/pctl_Cancelable.h>
#include <nn/pctl/detail/service/watcher/pctl_ServerResource.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherData.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

class TokenHolder;

////////////////////////////////////////////////////////////////////////////////

class OnlineCheckExecutor : public common::AsyncContext
{
public:
    explicit OnlineCheckExecutor(common::Cancelable* pCancelable) NN_NOEXCEPT :
        m_pCancelable(pCancelable)
    {
    }

    void SetParameters(ServerDeviceId deviceId) NN_NOEXCEPT
    {
        m_DeviceId = deviceId;
    }

    virtual void CloseContext() NN_NOEXCEPT NN_OVERRIDE
    {
        delete this;
    }

    virtual nn::Result Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsCancelTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCancelable->IsCanceled();
    }

private:

    common::Cancelable* m_pCancelable;
    ServerDeviceId m_DeviceId;
};

////////////////////////////////////////////////////////////////////////////////

class UpdateDeviceExecutor : public common::AsyncContext
{
public:
    explicit UpdateDeviceExecutor(common::Cancelable* pCancelable) NN_NOEXCEPT :
        m_pCancelable(pCancelable)
    {
    }

    void SetParameters(ServerDeviceId deviceId, const EtagInfo& etagSynchronized) NN_NOEXCEPT
    {
        m_DeviceId = deviceId;
        std::memcpy(&m_EtagSynchronized, &etagSynchronized, sizeof(EtagInfo));
    }

    virtual void CloseContext() NN_NOEXCEPT NN_OVERRIDE
    {
        delete this;
    }

    virtual nn::Result Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsCancelTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCancelable->IsCanceled();
    }

    static nn::Result ExecuteUpdateDevice(common::NetworkBuffer& bufferInfo,
        common::Cancelable* pCancelable,
        TokenHolder& tokenHolder,
        ServerDeviceId deviceId,
        const EtagInfo* etagSynchronized,
        nn::npns::NotificationToken* pNotificationTokenBuffer) NN_NOEXCEPT;

private:
    common::Cancelable* m_pCancelable;
    ServerDeviceId m_DeviceId;
    EtagInfo m_EtagSynchronized;
    nn::npns::NotificationToken m_NotificationToken;
};

////////////////////////////////////////////////////////////////////////////////

class UpdateDeviceAlarmSettingExecutor : public common::AsyncContext
{
public:
    explicit UpdateDeviceAlarmSettingExecutor(common::Cancelable* pCancelable) NN_NOEXCEPT :
        m_pCancelable(pCancelable)
    {
    }

    void SetParameters(ServerDeviceId deviceId, bool isDisabled, const nn::time::PosixTime& timeNow) NN_NOEXCEPT
    {
        m_DeviceId = deviceId;
        m_IsDisabled = isDisabled;
        m_TimeNow = timeNow;
    }

    virtual void CloseContext() NN_NOEXCEPT NN_OVERRIDE
    {
        delete this;
    }

    virtual nn::Result Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsCancelTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCancelable->IsCanceled();
    }

private:
    common::Cancelable* m_pCancelable;
    ServerDeviceId m_DeviceId;
    bool m_IsDisabled;
    nn::time::PosixTime m_TimeNow;
};

////////////////////////////////////////////////////////////////////////////////

}}}}}
