﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/ovln/format/ovln_PctlMessage.h>
#include <nn/pctl/detail/service/common/pctl_AsyncContext.h>
#include <nn/pctl/detail/service/common/pctl_Cancelable.h>
#include <nn/pctl/detail/service/system/pctl_Settings.h>
#include <nn/pctl/detail/service/watcher/pctl_ServerResource.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherData.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

////////////////////////////////////////////////////////////////////////////////

class RetrieveSettingsExecutor : public common::AsyncContext
{
public:
    explicit RetrieveSettingsExecutor(common::Cancelable* pCancelable) NN_NOEXCEPT
        : m_RetrieveParam()
    {
        m_RetrieveParam.pCancelable = pCancelable;
    }

    void SetParameters(const EtagInfo* etag, ServerDeviceId deviceId, int tryCount,
        bool isOverlayNotificationNecessary, bool forceRetrive) NN_NOEXCEPT
    {
        std::memcpy(&m_OriginalEtagInfo, etag, sizeof(EtagInfo));
        // 強制取得の場合は Etag を用いないようにする
        if (forceRetrive)
        {
            m_RetrieveParam.etagInfo.lastUpdatedAt = 0;
        }
        else
        {
            std::memcpy(&m_RetrieveParam.etagInfo, etag, sizeof(EtagInfo));
        }
        m_RetrieveParam.deviceId = deviceId;
        m_TryCount = tryCount;
        m_IsOverlayNotificationNecessary = isOverlayNotificationNecessary;
        m_RetrieveParam.exemptionSettings.Clear();
    }

    void SetParameters(ServerDeviceId deviceId,
        bool isOverlayNotificationNecessary,
        nn::ncm::ApplicationId applicationId, bool isExempt) NN_NOEXCEPT
    {
        m_RetrieveParam.etagInfo.lastUpdatedAt = 0;

        m_RetrieveParam.deviceId = deviceId;
        m_TryCount = 0;
        m_IsOverlayNotificationNecessary = isOverlayNotificationNecessary;

        m_RetrieveParam.exemptionSettings.Clear();
        if (applicationId != nn::ncm::ApplicationId::GetInvalidId())
        {
            m_RetrieveParam.exemptionSettings.AddExemptedValue(applicationId, isExempt);
        }
    }

    virtual void CloseContext() NN_NOEXCEPT NN_OVERRIDE
    {
        delete this;
    }

    virtual nn::Result Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsCancelTriggered() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_RetrieveParam.pCancelable->IsCanceled();
    }

public:
    struct RetrieveSettingsParamBuffer
    {
        // out
        nn::ovln::format::PctlSettingTypeFlagSet changedTypeFlags;
        // out
        bool isNotModified;
        // out
        bool isOnlyExemptionListUpdated;
        // in
        ServerDeviceId deviceId;
        // in,out
        common::NetworkBuffer* pBufferInfo;
        // in
        common::Cancelable* pCancelable;
        // in,out
        //   in:  isNotModified の判定に用いられる Etag を指定 (lastUpdatedAt が 0 の場合は使用しない)
        //   out: 取得した設定値の Etag (isNotModified が true の場合は空)
        EtagInfo etagInfo;
        // out
        system::Settings settings;
        // out
        system::FreeCommunicationApplicationSettings freeCommunicationSettings;
        // in,out
        system::ExemptApplicationSettings exemptionSettings;
        // out
        PlayTimerSettings playTimerSettings;
    };

    // @param[out] isExemptionListUpdated
    // @param[in, out] retrieveParam 処理に必要な追加データ
    static nn::Result ExecuteRetrieveAndStoreSettings(RetrieveSettingsParamBuffer& retrieveParam) NN_NOEXCEPT;

private:
    RetrieveSettingsParamBuffer m_RetrieveParam;
    EtagInfo m_OriginalEtagInfo;
    int m_TryCount;
    bool m_IsOverlayNotificationNecessary;
};

////////////////////////////////////////////////////////////////////////////////

}}}}}
