﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/npns/npns_Types.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkThread.h>
#include <nn/pctl/pctl_TypesWatcher.h>
#include <nn/pctl/detail/service/common/pctl_AsyncContext.h>
#include <nn/pctl/detail/service/common/pctl_NetworkBuffer.h>
#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>
#include <nn/pctl/detail/service/watcher/pctl_NotificationDataReceiver.h>
#include <nn/pctl/detail/service/watcher/pctl_ServerResource.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherData.h>
#include <atomic>

namespace nn { namespace nifm {
class NetworkConnection;
}}

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

/**
 * @brief Tokenを参照中であることを保持するHolderクラスです。
 */
class TokenHolder
{
    NN_DISALLOW_COPY(TokenHolder);

public:
    TokenHolder() NN_NOEXCEPT :
        m_Token(nullptr),
        m_pRefCount(nullptr)
    {
    }
    TokenHolder(TokenHolder&& holder) NN_NOEXCEPT
    {
        m_Token = std::move(holder.m_Token);
        m_pRefCount = std::move(holder.m_pRefCount);
    }
    ~TokenHolder() NN_NOEXCEPT
    {
        if (m_pRefCount != nullptr)
        {
            --(*m_pRefCount);
        }
    }
    TokenHolder& operator = (TokenHolder&& holder) NN_NOEXCEPT
    {
        m_Token = std::move(holder.m_Token);
        m_pRefCount = std::move(holder.m_pRefCount);
        return *this;
    }

    const char* GetToken() const NN_NOEXCEPT
    {
        return m_Token;
    }

    operator const char*() const NN_NOEXCEPT
    {
        return m_Token;
    }

private:
    void SetTokenData(const char* token, std::atomic_int* pRefCount) NN_NOEXCEPT
    {
        m_Token = token;
        m_pRefCount = pRefCount;
        ++(*pRefCount);
    }

    const char* m_Token;
    std::atomic_int* m_pRefCount;

    friend class NetworkManager;
};

class NetworkManager
{
public:
    static const size_t ThreadStackSize = 32 * 1024;         // curl処理で比較的スタックを消費
    static const size_t MaxMiiImageFileNameLength = 64;      // 64 > 4 ("mii/") + 16 (NA-ID) + 33 (Mii-FileName) + 4 (".png") + 2 (separator, null-char)
    static const size_t MaxMiiImageContentTypeLength = 32;   // マイナーな種別でなければ入るはず(strlen("image/png") == 9)

    NetworkManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT;
    ~NetworkManager() NN_NOEXCEPT;

    // @brief 設定ファイルを指定した変数に読み込みます。
    // @param[out] pOutRead 正しくデータが読み込めたかどうかを示す値を受け取るポインター
    // @param[out] outConfig 読み込んだデータを受け取る構造体へのインスタンス
    // @return nn::ResultSuccess またはfsに関するエラー
    // @details 読み込んだデータサイズが不一致の場合は config を 0 初期化の上、成功を返します。
    //    また、ファイルが見つからない場合はエラーではなく成功を返します。(その場合も config を 0 初期化します。)
    //    これらのケースの場合、*pOutRead は false になります。
    static nn::Result LoadConfig(bool* pOutRead, Config& outConfig) NN_NOEXCEPT;

    // @brief 設定ファイルを指定した変数に読み込みます。
    // @return nn::ResultSuccess またはfsに関するエラー
    // @details 読み込んだデータサイズが不一致の場合は config を 0 初期化の上、成功を返します。
    //    また、ファイルが見つからない場合はエラーではなく成功を返します。(その場合も config を 0 初期化します。)
    inline static nn::Result LoadConfig(Config& outConfig) NN_NOEXCEPT
    {
        return LoadConfig(nullptr, outConfig);
    }

    // @brief 指定した変数のデータを設定ファイルに書き込みます。
    // @return nn::ResultSuccess またはfsに関するエラー
    static nn::Result SaveConfig(const Config& config) NN_NOEXCEPT;

    // @brief config 内の連携情報を削除し、未連携状態にします。
    static void ClearPairingInfoFromConfig(Config& config) NN_NOEXCEPT;

    // @brief 設定ファイルを読み込みます。
    void LoadConfig() NN_NOEXCEPT;

    // @brief スレッドを開始します。
    // @details 終了はデストラクターで行います。
    void StartThread() NN_NOEXCEPT;

    // @brief Device Authentication Tokenを取得します。有効期限切れである場合は取得の通信処理を行います。
    nn::Result AcquireAuthenticationToken(TokenHolder& holder, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable) NN_NOEXCEPT;

    // @brief サーバーと連携済みかどうかを返します。
    bool IsPairingActive() const NN_NOEXCEPT
    {
        return m_Config.isPairingActive != 0;
    }

    // @brief 本体に保存されているサーバーデバイスIDを返します。
    ServerDeviceId GetSavedDeviceId() NN_NOEXCEPT
    {
        return m_Config.deviceId;
    }

    // @brief 取得したDevice Authentication Tokenを保存します。
    void StoreDeviceAuthenticationToken(const AuthenticationTokenData& token) NN_NOEXCEPT;

    // @brief 通知トークンの取得をこのセッションで行ったかどうかを返します。
    bool IsNotificationTokenRetrieved() const NN_NOEXCEPT
    {
        return m_IsNotificationTokenRetrieved;
    }

    // @brief 通知トークンを取得します。
    // @details HasNotificationToken() == false の場合の動作は未定義です。
    void GetNotificationToken(nn::npns::NotificationToken& token) NN_NOEXCEPT;

    // @brief 取得した通知トークンを保存します。
    // @details プロセスが稼働中の間、IsNotificationTokenRetrieved() の戻り値は true になります。
    void StoreNotificationToken(const nn::npns::NotificationToken& token) NN_NOEXCEPT;

    // @brief 連携した情報を保存します。連携済み状態にします。
    // @param[in] deviceId 連携先のサーバーデバイスID
    void StorePairingInfo(ServerDeviceId deviceId) NN_NOEXCEPT;

    // @brief 連携した情報を削除し、未連携状態にします。
    void ClearPairingInfo(bool notifyEvent) NN_NOEXCEPT;

    // @brief ペアコン設定同期用の Etag を取得します。
    void GetEtagForSettings(EtagInfo* outEtag) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(*outEtag) == sizeof(m_Config.etagForSettings));
        std::memcpy(outEtag, &m_Config.etagForSettings, sizeof(*outEtag));
    }

    // @brief ペアコン設定同期用の Etag を保存します。
    void StoreEtagForSettings(const EtagInfo* etagForSettings) NN_NOEXCEPT;

    // @brief デバイス情報を更新します。
    // @return 更新があれば true
    bool RefreshDeviceStatus() NN_NOEXCEPT;

    // @brief 内部で保持するデバイス情報を返します。
    void GetDeviceStatus(DeviceStatus& outDeviceStatus) NN_NOEXCEPT;

    // @brief PlayDataManager データにおけるアプリケーションイベントを最後に見た際の offset を返します。
    uint32_t GetLastPlayDataManagerAppEventOffset() NN_NOEXCEPT
    {
        return m_Config.lastPdmAppEventOffset;
    }

    // @brief PlayDataManager データにおけるアカウントイベントを最後に見た際の offset を返します。
    uint32_t GetLastPlayDataManagerAccountEventOffset() NN_NOEXCEPT
    {
        return m_Config.lastPdmAccountEventOffset;
    }

    // @brief PlayDataManager データの最終チェック情報を更新します。
    void UpdateLastPlayDataManagerOffsets(uint32_t appEventOffset, uint32_t accountEventOffset) NN_NOEXCEPT;

    // @brief 連携処理を開始します。outDeviceId・outActualOwnerCount・pOwners ポインターは処理終了まで有効である必要があります。
    nn::Result RequestPairing(common::AsyncContext** ppContext, ServerDeviceId* outDeviceId, size_t* outActualOwnerCount,
        const char* pairingCode, size_t codeLength, ServerDeviceOwner* pOwners, size_t maxOwnerCount) NN_NOEXCEPT;

    // @brief 連携処理を完了します。
    nn::Result RequestAuthorizePairing(common::AsyncContext** ppContext, ServerDeviceId deviceId) NN_NOEXCEPT;

    // @brief 連携処理を開始します。outActualOwnerCount・pOwners ポインターは処理終了まで有効である必要があります。
    nn::Result RequestRetrieveOwners(common::AsyncContext** ppContext, size_t* outActualOwnerCount,
        ServerDeviceOwner* pOwners, size_t maxOwnerCount, ServerDeviceId deviceId) NN_NOEXCEPT;

    // @brief バックグラウンドでオンラインチェック応答処理を行います。
    // @details Canceled が発生した場合は無視されます。
    void RequestOnlineCheckResponseBackground(ServerDeviceId deviceId) NN_NOEXCEPT;

    // @brief イベント送信処理を要求します。
    nn::Result RequestPostEvents(common::AsyncContext** ppContext, ServerDeviceId deviceId, size_t* outCount, EventData* pEventData, size_t eventDataSize) NN_NOEXCEPT;

    // @brief バックグラウンドでイベント送信処理を要求します。
    // @details Canceled が発生した場合は無視されます。
    void RequestPostEventsBackground(ServerDeviceId deviceId) NN_NOEXCEPT;

    // @brief バックグラウンドでイベント送信処理を要求します。
    // @details Canceled が発生した場合は無視されます。また、未連携の場合は何もしません。
    void RequestPostEventsBackground() NN_NOEXCEPT
    {
        if (IsPairingActive())
        {
            RequestPostEventsBackground(GetSavedDeviceId());
        }
    }

    // @brief バックグラウンドでデバイス情報更新処理を行います。
    // @details Canceled が発生した場合は無視されます。
    void RequestUpdateDeviceBackground() NN_NOEXCEPT;

    // @brief 連携解除処理を開始します。
    nn::Result RequestUnlinkDevice(common::AsyncContext** ppContext, ServerDeviceId deviceId, bool isAlwaysClearPairingInfo) NN_NOEXCEPT;

    // @brief ペアコン設定の同期をフォアグラウンド処理として開始します。取得した設定は直接適用されます。
    // @params[out] ppContext 非同期処理状況確認用のコンテキストを受け取るポインター(不要であれば nullptr)
    // @params[in] deviceId 設定取得に用いるサーバー上のデバイスID
    // @params[in] forceRetrieve Etag値によらずに受け取る場合は true
    inline nn::Result RequestSynchronizeSettingsForeground(common::AsyncContext** ppContext,
        ServerDeviceId deviceId,
        bool forceRetrieve) NN_NOEXCEPT
    {
        return RequestSynchronizeSettingsInternal(ppContext, deviceId, forceRetrieve, 0, false);
    }

    // @brief ペアコン設定の同期をバックグラウンド処理として開始します。取得した設定は直接適用されます。
    // @params[in] deviceId 設定取得に用いるサーバー上のデバイスID
    // @params[in] forceRetrieve Etag値によらずに受け取る場合は true
    // @parmas[in] tryCount 試行回数
    // @pre
    //  - tryCount > 0
    inline nn::Result RequestSynchronizeSettingsBackground(ServerDeviceId deviceId,
        bool forceRetrieve,
        int tryCount,
        bool fromNpns) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(tryCount, 0);
        return RequestSynchronizeSettingsInternal(nullptr, deviceId, forceRetrieve, tryCount, fromNpns);
    }

    // @brief 制限対象外リストの更新（およびペアコン設定の同期）処理を開始します。
    // @params[in] deviceId 設定取得に用いるサーバー上のデバイスID
    // @params[in] applicationId リストに追加するアプリケーションID
    // @params[in] isExempt applicationId で指定されるアプリケーションを制限対象外にする場合は true
    nn::Result RequestUpdateExemptionList(common::AsyncContext** ppContext,
        ServerDeviceId deviceId,
        nn::ncm::ApplicationId applicationId,
        bool isExempt) NN_NOEXCEPT;

    // @brief バックグラウンドでアラーム無効設定の更新処理を行います。
    // @details Canceled が発生した場合は無視されます。
    void RequestApplyAlarmSettingBackground(bool isDisabled) NN_NOEXCEPT;

    // @brief Mii画像処理を排他制御するためのMutexを返します。
    nn::os::SdkMutex& GetMutexForMiiImage() NN_NOEXCEPT
    {
        return m_MutexMiiImage;
    }

    // @brief ユーザーのMii画像を取得します。必要に応じて関数内でダウンロードを開始します。
    //     画像が見つからなかった場合、user.miiUri を空にする処理が行われます。
    //     ダウンロードが不要である場合、*ppContext に nullptr が返ります。
    // @details
    // - ダウンロードを開始した場合、FinishDownloadMiiImage の呼び出しが必要です。
    // - @ref GetMutexForMiiImage を利用した排他制御を行う必要があります。
    nn::Result TryToDownloadMiiImage(common::AsyncContext** ppContext, UserData& user) NN_NOEXCEPT;

    // @brief TryToDownloadMiiImage で開始したダウンロードを終了し、適切な戻り値を返します。
    // @details
    // @ref GetMutexForMiiImage を利用した排他制御を行う必要があります。
    nn::Result FinishDownloadMiiImage(common::AsyncContext* pContext, UserData& user) NN_NOEXCEPT;

    // @brief ユーザーのMii画像を取得します。必要に応じて関数内でダウンロードします。
    //     画像が見つからなかった場合、user.miiUri を空にする処理が行われます。
    // @details
    // @ref GetMutexForMiiImage を利用した排他制御を行う必要があります。
    nn::Result GetMiiImage(uint32_t* outActualSize, UserData& user, void* imageBuffer, size_t bufferSize) NN_NOEXCEPT;

    // @brief ユーザーのMii画像のファイル種別(Content-Type)を取得します。必要に応じて関数内でダウンロードします。
    //     画像が見つからなかった場合、user.miiUri を空にする処理が行われます。
    // @details
    // @ref GetMutexForMiiImage を利用した排他制御を行う必要があります。
    nn::Result GetMiiImageContentType(uint32_t* outActualLength, UserData& user, char* contentTypeStringBuffer, size_t maxLength) NN_NOEXCEPT;

    // @brief 間欠起動のバックグラウンド処理を開始します。
    nn::Result StartIntermittentOperation() NN_NOEXCEPT;

    // @brief 通信処理をキャンセルします。
    void Cancel() NN_NOEXCEPT
    {
        // バックグラウンド処理はキャンセルしない
        m_Cancelable.Cancel();
    }

    // @brief 間欠起動のバックグラウンド通信処理をキャンセルします。
    void CancelIntermittentOperation() NN_NOEXCEPT
    {
        m_CancelableForIntermittent.Cancel();
    }

    // @brief スリープに入る際に必要な処理を行います。
    void OnEnterSleep() NN_NOEXCEPT;

    // @brief スリープから復帰したときに必要な処理を行います。
    void OnAwakeFromSleep() NN_NOEXCEPT;

    // @brief ネットワーク処理を無効化します。
    // @details
    // 再起動するまで無効化は続き、プロセス起動時に
    // g_pMain->GetSettingsManager().IsAllFeaturesDisabled() が false となっている場合は
    // その無効化が持続します。
    //
    // また、このメソッドは内部で Cancel も行います。
    void DisableNetworkFeature() NN_NOEXCEPT;

    // @brief Watcherのネットワーク機能が無効化されているかどうかを返します。
    bool IsNetworkFeatureDisabled() const NN_NOEXCEPT;

    // @brief 設定値の最終更新日時を返します。
    // @param[out] outTime 更新日時を表す値
    // @return 最終更新日時があれば true、何らかの理由で未同期の場合は false
    bool GetSettingsUpdatedTime(nn::time::PosixTime* outTime) const NN_NOEXCEPT;

    // @brief サーバーとの通信エラーの中で連携が解除されているために起きるエラーについて
    //        連携情報削除などの後処理を行います。
    void HandlePairingDeletedStatus() NN_NOEXCEPT;

    // @brief 設定変更の通知を(後で)行うことをセットします。
    void SetNotifyNecessaryFlagForSettingChanged(nn::ovln::format::PctlSettingTypeFlagSet changedTypeFlags) NN_NOEXCEPT;

    // @brief 連携解除の通知を(後で)行うことをセットします。
    void SetNotifyNecessaryFlagForUnlinked(nn::ovln::format::PctlUnlinkReasonFlag reasonFlags) NN_NOEXCEPT;

    // @brief アラーム通知設定の変更通知を(後で)行うことをセットします。
    void SetNotifyNecessaryFlagForAlarmSettingChanged(nn::ovln::format::PctlAlarmSettingFlag flags) NN_NOEXCEPT;

    // @brief 最後に行ったデバイス情報アップロード処理が成功したかどうかを返します。
    // @details 本体起動時は false の状態となります。
    bool IsLastUpdateDeviceSuccess() const NN_NOEXCEPT
    {
        return m_IsLastUpdateDeviceSuccess;
    }

    // @brief 最後に行ったデバイス情報アップロード処理が成功したかどうかを設定します。
    void SetLastUpdateDeviceSuccess(bool isSuccess) NN_NOEXCEPT;

    // @brief 通信の結果によって連携解除が行われた際の通知用システムイベントを返します。
    nn::os::SystemEvent* GetUnlinkedEvent() NN_NOEXCEPT
    {
        return &m_SysEventUnlinked;
    }

    // @brief 連携が解除された通知(イベントのシグナル)とフラグをクリアします。
    // @details Watcher が利用できない環境下(メンテナンスモードなど)ではフラグのクリアのみ行います。
    static void ClearUnlinkedFlag() NN_NOEXCEPT;

    // @brief バックグラウンドタスクの終了を待機します。(テスト用)
    // @return バックグラウンドタスクの処理結果
    nn::Result WaitForBackgroundTaskForTest() NN_NOEXCEPT;

private:
    // @brief ペアコン設定の同期を開始します。取得した設定は直接適用されます。
    // @params[out] ppContext 非同期処理状況確認用のコンテキストを受け取るポインター(backgroundTryCount > 0 の場合、または不要であれば nullptr)
    // @params[in] deviceId 設定取得に用いるサーバー上のデバイスID
    // @params[in] forceRetrieve Etag値によらずに受け取る場合は true
    // @params[in] backgroundTryCount バックグラウンド通信である場合は試行回数、フォアグラウンド通信である場合は 0 を指定
    // @params[in] fromNpns NPNS通知由来の処理であれば true
    // @details backgroundTryCount > 0 の場合は *ppContext のインスタンスが設定されません。
    nn::Result RequestSynchronizeSettingsInternal(common::AsyncContext** ppContext,
        ServerDeviceId deviceId,
        bool forceRetrieve,
        int backgroundTryCount,
        bool fromNpns) NN_NOEXCEPT;

    nn::Result GetMiiImageInternal(uint32_t* outActualSize, uint32_t* outActualContentTypeLength,
        UserData& user, void* imageBuffer, size_t bufferSize,
        char* contentTypeStringBuffer, size_t contentTypeMaxLength) NN_NOEXCEPT;

    void LoadConfigNoLock() NN_NOEXCEPT;
    void LoadDeviceStatusNoLock() NN_NOEXCEPT;
    void SaveConfigNoLock() NN_NOEXCEPT;

    static nn::Result LoadDeviceStatus(bool* pOutRead, DeviceStatus& outDeviceStatus) NN_NOEXCEPT;
    static nn::Result SaveDeviceStatus(const DeviceStatus& deviceStatus) NN_NOEXCEPT;

    void ClearTokensIfEnvironmentChanged() NN_NOEXCEPT;

    nn::Result Connect(nn::nifm::NetworkConnection& connection, common::AsyncContext* pContext) NN_NOEXCEPT;
    void ExecuteContext(common::AsyncContext* pContext) NN_NOEXCEPT;
    void ExecuteContext(common::AsyncContext* pContext, common::NetworkBuffer& buffer) NN_NOEXCEPT;

    // @brief 次の AsyncContext を設定します。既に設定されている場合は待機します。
    // @return 設定できたら nn::ResultSuccess、待機中に終了イベントが発生したら nn::pctl::ResultCanceled
    // @details nn::ResultSuccess 以外の場合は pContext->CloseContext() を呼び出します。
    nn::Result SetNextAsyncContext(common::AsyncContext* pContext) NN_NOEXCEPT;

    void HandleFinishedBackgroundTask(common::AsyncContext* pContext) NN_NOEXCEPT;

    // @brief 実行中のバックグラウンドタスクがある場合、それをキャンセルして遅延させます。
    void PostponeBackgroundTask() NN_NOEXCEPT;

    // @brief m_Config.notifyNecessaryFlags の値をチェックして必要なら通知を行います。
    void CheckNotifyNecessaryFlags() NN_NOEXCEPT;

    void ThreadProc() NN_NOEXCEPT;
    static void StaticThreadProc(void* pThis) NN_NOEXCEPT
    {
        reinterpret_cast<NetworkManager*>(pThis)->ThreadProc();
    }

    template <typename TEvent>
    class EventHolderT
    {
    public:
        EventHolderT() NN_NOEXCEPT :
            m_Event(nn::os::EventClearMode::EventClearMode_ManualClear)
        {
            nn::os::InitializeMultiWaitHolder(&m_WaitHolder, m_Event.GetBase());
        }
        ~EventHolderT() NN_NOEXCEPT
        {
            nn::os::FinalizeMultiWaitHolder(&m_WaitHolder);
        }

        TEvent m_Event;
        nn::os::MultiWaitHolderType m_WaitHolder;
    };
    typedef EventHolderT<nn::os::Event> EventHolder;

    class ExternalSystemEventHolder
    {
    public:
        ExternalSystemEventHolder() NN_NOEXCEPT :
            m_Event(),
            m_WaitHolder(),
            m_IsInitialized(false)
        {
        }

        ~ExternalSystemEventHolder() NN_NOEXCEPT
        {
            if (m_IsInitialized)
            {
                //m_pEventBase = nullptr;
                nn::os::FinalizeMultiWaitHolder(&m_WaitHolder);
            }
        }

        void Initialize() NN_NOEXCEPT
        {
            m_IsInitialized = true;
            nn::os::InitializeMultiWaitHolder(&m_WaitHolder, m_Event.GetBase());
        }

        nn::os::SystemEvent m_Event;
        nn::os::MultiWaitHolderType m_WaitHolder;
        bool m_IsInitialized;
    };

    nn::os::ThreadType m_TaskThread;
    nn::os::MultiWaitType m_MultiWait;
    EventHolder m_EventHolderExit;
    EventHolder m_EventHolderMessage;
    ExternalSystemEventHolder m_EventHolderNpns;
    nn::os::SystemEvent m_SysEventUnlinked;
    mutable nn::os::SdkMutex m_MutexMessage;
    mutable nn::os::SdkMutex m_MutexSaveData;
    mutable nn::os::SdkMutex m_MutexToken;
    mutable nn::os::SdkMutex m_MutexMiiImage;
    mutable nn::os::SdkMutex m_MutexBackgroundTask;
    common::AsyncContext* m_pNextAsyncContext;
    common::AsyncContext* m_pRunningBackgroundContext;
    common::AsyncContext* m_pNextBackgroundAsyncContext;
    nn::Result m_LastBackgroundTaskResult;
    nn::npns::NotificationToken m_NotificationToken;
    bool m_IsThreadRunning;
    bool m_IsNetworkDisabledOnInit;
    bool m_IsBackgroundCanceled;
    bool m_IsNotificationTokenRetrieved;
    bool m_IsLastUpdateDeviceSuccess;
    bool m_IsOverlayDisabled;
    std::atomic_int m_TokenRefCount;
    Config m_Config;
    DeviceStatus m_DeviceStatus;
    NotificationDataReceiver m_NotificationDataReceiver;
    common::Cancelable m_Cancelable; //!< 通常通信(API呼び出しに伴うものなど)のキャンセル用
    common::Cancelable m_CancelableForBackground; //!< バックグラウンド通信のキャンセル用
    common::Cancelable m_CancelableForIntermittent; //!< 間欠起動中タスクのキャンセル用
    common::NetworkBuffer* m_pBuffer; //!< 原則 ThreadProc からのみ利用
    char* m_MiiImageFileName; //!< ダウンロード済みMii画像のファイル名
    char* m_MiiImageFileNameBuffer; //!< ファイル名生成用の一時バッファー
    char* m_MiiImageContentType; //!< ダウンロード済みMii画像のContent-Type

    NN_STATIC_ASSERT((std::is_same<decltype(m_MutexMessage), decltype(m_MutexSaveData)>::value));
    NN_STATIC_ASSERT((std::is_same<decltype(m_MutexMessage), decltype(m_MutexToken)>::value));
};

}}}}}
