﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/pctl/pctl_TypesSystem.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace system {

static const size_t MinPinCodeLength = 4;
static const size_t MaxPinCodeLength = 8;

// (1 は欠番)
static const uint8_t SettingsDataVersion2 = 2;
static const uint8_t SettingsDataVersionCurrent = SettingsDataVersion2;

enum SizedSafetyLevel : uint8_t
{
    SizedSafetyLevel_None = SafetyLevel_None,
    SizedSafetyLevel_Custom = SafetyLevel_Custom,
    SizedSafetyLevel_Children = SafetyLevel_Children,
    SizedSafetyLevel_YoungTeens = SafetyLevel_YoungTeens,
    SizedSafetyLevel_OlderTeens = SafetyLevel_OlderTeens,
};

namespace old
{
    struct CurrentSettingsVer1
    {
        char pinCode[MaxPinCodeLength];                   //!< 解除コード(8文字ちょうどの場合終端のNULL文字無し)
        SizedSafetyLevel safetyLevel;                     //!< 安心レベル
        uint8_t ratingAge;                                //!< レーティング制限に使用する制限年齢
        nn::ns::RatingOrganization ratingOrganization;    //!< レーティング団体
        uint8_t isDefaultRatingOrganizationSet : 1;       //!< 既定のレーティング団体がセットされているかどうか
        uint8_t isSnsPostRestricted : 1;                  //!< SNS投稿が制限されるかどうか
        uint8_t isFreeCommunicationRestrictedByDefault : 1; //!< 既定で自由なコミュニケーションが制限されるかどうか
        uint8_t : 5;                                      //!< (padding)
    };
    struct CustomSettingsVer1
    {
        uint8_t ratingAge;                                //!< レーティング制限に使用する制限年齢
        uint8_t isSnsPostRestricted : 1;                  //!< SNS投稿が制限されるかどうか
        uint8_t isFreeCommunicationRestrictedByDefault : 1; //!< 既定で自由なコミュニケーションが制限されるかどうか
        uint8_t : 6;                                      //!< (padding)
    };
}

/**
 * @brief 永続化されるペアコン基本設定の現在の設定値情報です。
 * @details
 * 各値は「安心レベル」に基づく既定値からセットされる場合もありますが、
 * 制限判定時は原則としてこの構造体の値を使用します。
 */
struct CurrentSettings
{
    char pinCode[MaxPinCodeLength];                   //!< 解除コード(8文字ちょうどの場合終端のNULL文字無し)
    SizedSafetyLevel safetyLevel;                     //!< 安心レベル
    uint8_t ratingAge;                                //!< レーティング制限に使用する制限年齢
    nn::ns::RatingOrganization ratingOrganization;    //!< レーティング団体
    uint8_t isDefaultRatingOrganizationSet : 1;       //!< 既定のレーティング団体がセットされているかどうか
    uint8_t isSnsPostRestricted : 1;                  //!< SNS投稿が制限されるかどうか
    uint8_t isFreeCommunicationRestrictedByDefault : 1; //!< 既定で自由なコミュニケーションが制限されるかどうか
    uint8_t isStereoVisionRestricted : 1;             //!< [Ver2以降] 立体視機能が制限されているか
    uint8_t _padding1 : 4;                            //!< (padding)
    uint8_t dataVersion;                              //!< データバージョン(Ver1の場合はこのフィールドとそれ以降が存在しない)
    uint8_t _padding2[3];                             //!< (padding)
};
NN_STATIC_ASSERT(sizeof(CurrentSettings) > sizeof(old::CurrentSettingsVer1));

/**
 * @brief 「安心レベル」が「カスタム」のときに使用されるペアコン設定の情報です。
 */
struct CustomSettings
{
    uint8_t ratingAge;                                //!< レーティング制限に使用する制限年齢
    uint8_t isSnsPostRestricted : 1;                  //!< SNS投稿が制限されるかどうか
    uint8_t isFreeCommunicationRestrictedByDefault : 1; //!< 既定で自由なコミュニケーションが制限されるかどうか
    uint8_t _padding : 6;                             //!< (padding)
    // MEMO: Ver1 では _padding の部分がフィールドなしになっていたが、Ver2 からは明示的に存在するフィールドとする
};

namespace old
{
    struct SettingsVer1
    {
        CurrentSettingsVer1 current;
        CustomSettingsVer1 custom;
    };
}

/**
 * @brief 永続化されるペアコン基本設定情報です。
 */
struct Settings
{
    CurrentSettings current;
    CustomSettings custom;

    void Reset() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
        current.dataVersion = SettingsDataVersionCurrent;
        current.ratingOrganization = nn::ns::RatingOrganization::CERO;
    }
};
NN_STATIC_ASSERT(sizeof(Settings) > sizeof(old::SettingsVer1));

struct FreeCommunicationApplicationSettings
{
    struct Partial
    {
        //!< 各アプリケーションに対する制限値を格納する型
        typedef uint32_t RestrictFlagType;
        //!< この構造体で管理するアプリケーションIDの数
        static const int IdCountPerStructure = 32;
        NN_STATIC_ASSERT(static_cast<size_t>(IdCountPerStructure) == sizeof(RestrictFlagType) * 8);

        //!< 各アプリケーションに対する制限が有効かどうかを管理する値
        RestrictFlagType isRestrictedFlags;
        //!< 比較処理のマーク付けに利用する値
        uint32_t compareCheckFlags;
        NN_STATIC_ASSERT(static_cast<size_t>(IdCountPerStructure) == sizeof(uint32_t) * 8);
        //!< 個々のアプリケーションIDを持つ配列(nn::ncm::ApplicationId::GetInvalidId() とそれ以降の要素は無効)
        nn::ncm::ApplicationId applicationIds[IdCountPerStructure];

        bool IsRestricted(int index) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            return (isRestrictedFlags & (1ul << index)) != 0;
        }
        void SetRestricted(int index, bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            if (value)
            {
                isRestrictedFlags |= (1ul << index);
            }
            else
            {
                isRestrictedFlags &= ~(1ul << index);
            }
        }
        bool IsCompared(int index) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            return (compareCheckFlags & (1ul << index)) == 0;
        }
        void SetCompared(int index, bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            if (value)
            {
                compareCheckFlags &= ~(1ul << index);
            }
            else
            {
                compareCheckFlags |= (1ul << index);
            }
        }
        bool HasAnyItemNotCompared(int totalCount) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_MINMAX(totalCount, 1, (std::integral_constant<int, IdCountPerStructure>::value));
            return (compareCheckFlags & ((1ul << totalCount) - 1)) != 0;
        }
        void SetAllItemNotCompared() NN_NOEXCEPT
        {
            compareCheckFlags = ~static_cast<decltype(compareCheckFlags)>(0);
        }
    };

    // Partial 構造体の配列の要素数
    static const size_t PartialArrayCount = 93;
    // アプリケーションの数を 2000 以上格納できるようにする
    // (2000: メニュー上での最大アプリケーション数の想定)
    NN_STATIC_ASSERT(static_cast<size_t>(Partial::IdCountPerStructure) * PartialArrayCount >= 2000);

    Partial data[PartialArrayCount];

    // @brief 有効なデータの数を返します。
    size_t GetAvailableCount() const NN_NOEXCEPT;
    // @brief 有効なデータのうち1つでも制限が有効であるものがあれば true を返します。
    bool IsAnyRestrictionEnabled() const NN_NOEXCEPT;
    // @brief ApplicationIdをキーとして制限設定値を取得します。
    // @return id がデータ内に見つかったら true
    bool GetRestrictedValueByApplicationId(bool* outValue, nn::ncm::ApplicationId id) const NN_NOEXCEPT;
    // @brief 設定値データを取得し配列に格納します。
    size_t GetRestrictedValues(nn::pctl::FreeCommunicationApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT;
    // @brief info内のApplicationIdをキーとして値を設定します。
    // @return id がデータ内に見つかったら true
    bool SetRestrictedValueByApplicationId(const nn::pctl::FreeCommunicationApplicationInfo& info) NN_NOEXCEPT;
    // @brief ApplicationIdを追加します。追加された場合は true、既に存在しているなど追加されなかった場合は false を返します。
    bool AddRestrictedValue(nn::ncm::ApplicationId id, bool defaultValue) NN_NOEXCEPT;
    // @brief 特定のタイトルをリストから削除します。(主にデバッグ用)
    // @return id がデータ内に見つかったら true
    bool DeleteRestrictedValueByApplicationId(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief リストの設定値をリセットします。リスト自体はそのままになります。
    void ResetValues() NN_NOEXCEPT;
    // @brief リストを空にします。
    void Clear() NN_NOEXCEPT;
    // @brief 引数と保持するデータ・設定値が同じかどうかを確認しつつ、保持しているデータの設定値の更新を行います。
    // @param[in] list 更新する設定値を持つリスト
    // @param[in] unknownTitleCallback 「list にあるが自データにないタイトル」があった際に呼び出されるコールバック関数
    // @param[in] notTitleInListCallback 「自データにはあるが list にないタイトル」があった際に呼び出されるコールバック関数
    // @return 「設定値の書き換え」があれば true
    // @details
    //  - コールバック関数の引数に nullptr を指定するとそれらは呼び出されません。
    //  - 互いのリストにないタイトルがあっても(コールバック関数が呼び出されても)、
    //    リストにあるタイトル同士の設定値がすべて一致していた場合は戻り値は false になります。
    bool UpdateDataFromList(const FreeCommunicationApplicationSettings& list,
        void (* unknownTitleCallback)(nn::ncm::ApplicationId id),
        void (* notTitleInListCallback)(nn::ncm::ApplicationId id)) NN_NOEXCEPT;
};
// サイズは約 24KiB (24KiB を超えない)とする
NN_STATIC_ASSERT(sizeof(FreeCommunicationApplicationSettings) <= 24 * 1024);

struct ExemptApplicationSettings
{
    struct Partial
    {
        //!< 各アプリケーションに対する制限対象外値を格納する型
        typedef uint32_t ExemptedFlagType;
        //!< この構造体で管理するアプリケーションIDの数
        static const int IdCountPerStructure = 32;
        NN_STATIC_ASSERT(static_cast<size_t>(IdCountPerStructure) == sizeof(ExemptedFlagType) * 8);

        //!< 各アプリケーション制限対象外かどうかを管理する値
        ExemptedFlagType isExemptedFlags;
        //!< 比較処理のマーク付けに利用する値
        uint32_t compareCheckFlags;
        NN_STATIC_ASSERT(static_cast<size_t>(IdCountPerStructure) == sizeof(uint32_t) * 8);
        //!< 個々のアプリケーションIDを持つ配列(nn::ncm::ApplicationId::GetInvalidId() とそれ以降の要素は無効)
        nn::ncm::ApplicationId applicationIds[IdCountPerStructure];

        bool IsExempted(int index) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            return (isExemptedFlags & (1ul << index)) != 0;
        }
        void SetExempted(int index, bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            if (value)
            {
                isExemptedFlags |= (1ul << index);
            }
            else
            {
                isExemptedFlags &= ~(1ul << index);
            }
        }
        bool IsCompared(int index) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            return (compareCheckFlags & (1ul << index)) == 0;
        }
        void SetCompared(int index, bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(index, 0, (std::integral_constant<int, IdCountPerStructure>::value));
            if (value)
            {
                compareCheckFlags &= ~(1ul << index);
            }
            else
            {
                compareCheckFlags |= (1ul << index);
            }
        }
        bool HasAnyItemNotCompared(int totalCount) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_MINMAX(totalCount, 1, (std::integral_constant<int, IdCountPerStructure>::value));
            return (compareCheckFlags & ((1ul << totalCount) - 1)) != 0;
        }
        void SetAllItemNotCompared() NN_NOEXCEPT
        {
            compareCheckFlags = ~static_cast<decltype(compareCheckFlags)>(0);
        }
    };

    // Partial 構造体の配列の要素数
    static const size_t PartialArrayCount = 93;
    // アプリケーションの数を 2000 以上格納できるようにする
    // (2000: メニュー上での最大アプリケーション数の想定)
    NN_STATIC_ASSERT(static_cast<size_t>(Partial::IdCountPerStructure) * PartialArrayCount >= 2000);

    Partial data[PartialArrayCount];

    // @brief 有効なデータの数を返します。
    size_t GetAvailableCount() const NN_NOEXCEPT;
    // @brief 有効なデータのうち1つでも制限対象外であるものがあれば true を返します。
    bool IsAnyExemptionEnabled() const NN_NOEXCEPT;
    // @brief ApplicationIdをキーとして制限設定値を取得します。
    // @return id がデータ内に見つかったら true
    bool GetExemptedValueByApplicationId(bool* outValue, nn::ncm::ApplicationId id) const NN_NOEXCEPT;
    // @brief 設定値データを取得し配列に格納します。
    size_t GetExemptedValues(nn::pctl::ExemptApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT;
    // @brief info内のApplicationIdをキーとして値を設定します。
    // @return id がデータ内に見つかったら true
    bool SetExemptedValueByApplicationId(const nn::pctl::ExemptApplicationInfo& info) NN_NOEXCEPT;
    // @brief ApplicationIdを追加します。追加された場合は true、既に存在しているなど追加されなかった場合は false を返します。
    bool AddExemptedValue(nn::ncm::ApplicationId id, bool defaultValue) NN_NOEXCEPT;
    // @brief 特定のタイトルをリストから削除します。(主にデバッグ用)
    // @return id がデータ内に見つかったら true
    bool DeleteExemptedValueByApplicationId(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief リストの設定値をリセットします。リスト自体はそのままになります。
    void ResetValues() NN_NOEXCEPT;
    // @brief リストを空にします。
    void Clear() NN_NOEXCEPT;
    // @brief 引数と保持するデータ・設定値が同じかどうかを確認しつつ、保持しているデータの設定値の更新を行います。
    // @param[in] list 更新する設定値を持つリスト
    // @param[in] unknownTitleCallback 「list にあるが自データにないタイトル」があった際に呼び出されるコールバック関数
    // @param[in] notTitleInListCallback 「自データにはあるが list にないタイトル」があった際に呼び出されるコールバック関数
    // @return 「設定値の書き換え」があれば true
    // @details
    //  - コールバック関数の引数に nullptr を指定するとそれらは呼び出されません。
    //  - 互いのリストにないタイトルがあっても(コールバック関数が呼び出されても)、
    //    リストにあるタイトル同士の設定値がすべて一致していた場合は戻り値は false になります。
    bool UpdateDataFromList(const ExemptApplicationSettings& list,
        void(*unknownTitleCallback)(nn::ncm::ApplicationId id),
        void(*notTitleInListCallback)(nn::ncm::ApplicationId id)) NN_NOEXCEPT;
    // @brief
    bool GetExemptedValueByIndex(nn::pctl::ExemptApplicationInfo* pOutInfo, size_t index) const NN_NOEXCEPT;
};
// サイズは約 24KiB (24KiB を超えない)とする
NN_STATIC_ASSERT(sizeof(ExemptApplicationSettings) <= 24 * 1024);

}}}}}
