﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pctl/detail/service/watcher/pctl_NetworkManager.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherEventManager.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherEventStorage.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace pctl { namespace detail { namespace service {

struct PairingInfoData
{
    watcher::ServerDeviceId deviceId;
    watcher::ServerDeviceOwner* arrayOwners; // service::AllocateMemoryBlock で割り当てる
    size_t countOwners;
    uint64_t lastTickToUse;
};

class PairingInfoDataHolder
{
    NN_DISALLOW_COPY(PairingInfoDataHolder);

public:
    PairingInfoDataHolder() NN_NOEXCEPT :
        m_pMutex(nullptr),
        m_pData(nullptr)
    {
    }

    PairingInfoDataHolder(PairingInfoDataHolder&& holder) NN_NOEXCEPT :
        m_pMutex(holder.m_pMutex),
        m_pData(holder.m_pData)
    {
        holder.m_pMutex = nullptr;
        holder.m_pData = nullptr;
    }

    ~PairingInfoDataHolder() NN_NOEXCEPT
    {
        if (m_pMutex != nullptr)
        {
            m_pMutex->Unlock();
        }
    }

    PairingInfoData* operator -> () NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pData);
        return m_pData;
    }

    PairingInfoDataHolder& operator = (PairingInfoDataHolder&& holder) NN_NOEXCEPT
    {
        m_pMutex = holder.m_pMutex;
        m_pData = holder.m_pData;
        holder.m_pMutex = nullptr;
        holder.m_pData = nullptr;
        return *this;
    }

private:
    explicit PairingInfoDataHolder(nn::os::SdkMutex* pMutex, PairingInfoData* pData) NN_NOEXCEPT :
        m_pMutex(pMutex),
        m_pData(pData)
    {
        if (m_pMutex != nullptr)
        {
            m_pMutex->lock();
        }
    }

    friend class ServiceWatcher;

    nn::os::SdkMutex* m_pMutex;
    PairingInfoData* m_pData;
};

class ServiceWatcher
{
public:
    ServiceWatcher() NN_NOEXCEPT;
    ~ServiceWatcher() NN_NOEXCEPT;

    watcher::NetworkManager& GetNetworkManager() NN_NOEXCEPT
    {
        return m_NetworkManager;
    }

    watcher::WatcherEventManager& GetWatcherEventManager() NN_NOEXCEPT
    {
        return m_WatcherEventManager;
    }

    watcher::WatcherEventStorage& GetWatcherEventStorage() NN_NOEXCEPT
    {
        return m_WatcherEventStorage;
    }

    void InitializeAndStartThread() NN_NOEXCEPT;

    PairingInfoDataHolder AcquirePairingInfoData() NN_NOEXCEPT
    {
        return PairingInfoDataHolder(&m_MutexPairingInfoData, &m_PairingInfoData);
    }

private:
    watcher::NetworkManager m_NetworkManager;
    watcher::WatcherEventManager m_WatcherEventManager;
    watcher::WatcherEventStorage m_WatcherEventStorage;
    nn::os::SdkMutex m_MutexPairingInfoData;
    PairingInfoData m_PairingInfoData;
};

/**
 * @brief Watcherの初期化を行います。
 */
void InitializeWatcher() NN_NOEXCEPT;
/**
 * @brief Watcherの終了処理を行います。
 * @details InitializeWatcher が呼び出されていない場合は何もしません。
 */
void FinalizeWatcher() NN_NOEXCEPT;
/**
 * @brief Watcherが初期化済みで利用可能かどうかを返します。
 */
bool IsWatcherAvailable() NN_NOEXCEPT;

// @brief サービス内で共通のグローバルデータを持つインスタンスです。
extern ServiceWatcher* g_pWatcher;

}}}}
