﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/detail/service/json/pctl_Json.h>
#include <nn/pctl/detail/service/json/pctl_JsonDataHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/common/pctl_Cancelable.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

typedef nn::Result (* HandleWebErrorFunction)(void* param, int statusCode, nn::Result defaultResult);

class JsonWebErrorDataHandler : public JsonDataHandler
{
public:
    JsonWebErrorDataHandler(void* param, const ExpectDataDefinition* dataArray, int dataCount) NN_NOEXCEPT :
        JsonDataHandler(param, dataArray, dataCount),
        m_ErrorCallback(nullptr)
    {
    }

    template <size_t N>
    JsonWebErrorDataHandler(void* param, const ExpectDataDefinition(&dataArray)[N]) NN_NOEXCEPT :
        JsonDataHandler(param, dataArray),
        m_ErrorCallback(nullptr)
    {
    }

    JsonWebErrorDataHandler(void* param, HandleWebErrorFunction errorCallback, const ExpectDataDefinition* dataArray, int dataCount) NN_NOEXCEPT :
        JsonDataHandler(param, dataArray, dataCount),
        m_ErrorCallback(errorCallback)
    {
    }

    template <size_t N>
    JsonWebErrorDataHandler(void* param, HandleWebErrorFunction errorCallback, const ExpectDataDefinition(&dataArray)[N]) NN_NOEXCEPT :
        JsonDataHandler(param, dataArray),
        m_ErrorCallback(errorCallback)
    {
    }

    nn::Result CallErrorFunction(int statusCode, nn::Result parseResult) NN_NOEXCEPT;

private:
    HandleWebErrorFunction m_ErrorCallback;
};

/*!
    @brief      発行されたHTTPリクエストに対して受信したJSONデータをパースします。

    @param[in]  pHandler              データハンドラー。
    @param[in]  pErrorDataHandler     サーバーエラー時のデータハンドラー。nullptr の場合はエラー時に解析しません。
    @param[in]  pStream               入力ストリーム。
    @param[in]  jsonValueBuffer       JSONの値を一時的にキープするバッファー。
    @param[in]  jsonValueBufferSize   jsonValueBuffer の最大サイズ。
    @param[in]  streamWorkBuffer      JSONの値を一時的にキープするバッファー。
    @param[in]  streamWorkBufferSize  streamWorkBuffer の最大サイズ。
    @param[in]  pCancelable           キャンセル可能オブジェクト。

    @return     処理結果。
*/
nn::Result ParseWebStream(JsonDataHandler* pHandler,
    JsonWebErrorDataHandler* pErrorDataHandler,
    JsonHttpInputStream* pStream,
    void* jsonValueBuffer,
    size_t jsonValueBufferSize,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT;

/*!
    @brief      発行されたHTTPリクエストに対してJSONデータをPOSTし、応答として受信したJSONデータをパースします。

    @param[in]  pWriter               POSTするJSONデータを生成する JsonStructuredWriter へのインスタンス
    @param[in]  pHandler              データハンドラー。
    @param[in]  pErrorDataHandler     サーバーエラー時のデータハンドラー。nullptr の場合はエラー時に解析しません。
    @param[in]  pStream               入力ストリーム。
    @param[in]  jsonValueBuffer       JSONの値を一時的にキープするバッファー。
    @param[in]  jsonValueBufferSize   jsonValueBuffer の最大サイズ。
    @param[in]  streamWorkBuffer      JSONの値を一時的にキープするバッファー。
    @param[in]  streamWorkBufferSize  streamWorkBuffer の最大サイズ。
    @param[in]  pCancelable           キャンセル可能オブジェクト。

    @return     処理結果。
*/
nn::Result ParseWebStreamWithPostJSON(JsonStructuredWriter* pWriter,
    JsonDataHandler* pHandler,
    JsonWebErrorDataHandler* pErrorDataHandler,
    JsonHttpInputStream* pStream,
    void* jsonValueBuffer,
    size_t jsonValueBufferSize,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT;

}}}}}
