﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

/*!
    @brief      メモリ上にデータを書き込む出力ストリームです。
*/
class JsonMemoryOutputStream
{
public:
    typedef char Ch;

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonMemoryOutputStream() NN_NOEXCEPT :
        m_Memory(nullptr),
        m_MemorySize(0),
        m_FilledSize(0),
        m_IsOpended(false)
    {
    }

    /*!
        @brief      出力ストリームを開きます。

        @param[in]  memory  出力データ。
        @param[in]  size    出力データのサイズ。

        @pre
            - memory != nullptr
            - size > 0

        @return     処理結果。
    */
    void Open(Ch* memory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(memory);
        NN_SDK_REQUIRES(size > 0);

        NN_SDK_ASSERT(!m_IsOpended);

        m_Memory = memory;
        m_MemorySize = size;
        m_FilledSize = 0;

        m_IsOpended = true;
    }

    /*!
        @brief      出力ストリームを閉じます。
    */
    void Close() NN_NOEXCEPT
    {
        m_IsOpended = false;
    }

    /*!
        @brief      データの入力を開始します。
    */
    void PutBegin() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsOpended);

        m_FilledSize = 0;
    }

    /*!
        @brief      データを入力します。

        @param[in]  c   文字。
    */
    void Put(Ch c) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsOpended);
        NN_SDK_ASSERT(m_FilledSize < m_MemorySize);

        m_Memory[m_FilledSize++] = c;
    }

    /*!
        @brief      データの入力を終了します。

        @return     入力した文字数。
    */
    size_t PutEnd() NN_NOEXCEPT
    {
        if (m_FilledSize < m_MemorySize)
        {
            m_Memory[m_FilledSize] = '\0';
        }

        return m_FilledSize;
    }

    void Flush() NN_NOEXCEPT
    {
    }

private:
    //
    Ch* m_Memory;
    size_t m_MemorySize;
    //
    size_t m_FilledSize;
    //
    bool m_IsOpended;
};

}}}}}
