﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/pctl/detail/service/common/pctl_Cancelable.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

class HttpRequest;

// @brief HTTPステータスコードを nn::Result 値に変換します。
nn::Result ConvertStatusCodeToResult(int statusCode) NN_NOEXCEPT;

// @brief HttpRequest を用いた、bodyレスポンスを見ないシンプルなリクエストを発行します。
nn::Result PerformRequest(HttpRequest* pRequest,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT;

class HttpRequest
{
public:
    /*!
        @brief      コンストラクタです。
    */
    HttpRequest() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~HttpRequest() NN_NOEXCEPT;

    /*!
        @brief      リクエストヘッダ 1 行のバッファサイズです。
    */
    static const size_t RequestHeaderLineSize = 2048;

    /*!
        @brief      ヘッダの受信コールバックです。

                    処理を中断したい場合、false を返してください。
    */
    typedef bool (* ResponseHeaderCallback)(char* label, char* value, void* param);

    /*!
        @brief      ヘッダの受信完了コールバックです。

                    処理を中断したい場合、false を返してください。
    */
    typedef bool (* ResponseHeaderEndCallback)(int statusCode, void* param);

    /*!
        @brief      ヘッダのデータ送信コールバックです。

                    0 を返すとデータの終了を示すことができます。
    */
    typedef size_t (* HttpSendDataCallback)(char* buffer, size_t size, size_t count, void* param);
    /*!
        @brief      ヘッダのデータ受信コールバックです。
    */
    typedef size_t (* HttpReceiveDataCallback)(char* buffer, size_t size, size_t count, void* param);

    /*!
        @brief      入力ストリームを開きます。

        @param[in]  url URL。

        @pre
            - url != nullptr

        @return     処理結果。

        @details
                    本関数は、 GET リクエストを発行します。
    */
    nn::Result Open(const char* url) NN_NOEXCEPT;

    /*!
        @brief      入力ストリームを開きます。

        @param[in]  request リクエスト。（GET / PATCH / etc...）
        @param[in]  url     URL。

        @pre
            - request != nullptr
            - url != nullptr

        @return     処理結果。
    */
    nn::Result Open(const char* request, const char* url) NN_NOEXCEPT;

    /*!
        @brief      通信を閉じます。
    */
    void Close() NN_NOEXCEPT;

    /*!
        @brief      リクエストヘッダを追加します。

        @param[in]  header  ヘッダ情報(「XXX: yyy」の形式)。

        @return     処理結果。
    */
    nn::Result AddRequestHeader(const char* header) NN_NOEXCEPT;

    /*!
        @brief      リクエストヘッダを追加します。

        @param[in]  format  フォーマット。
        @param[in]  ...     可変長引数。

        @return     処理結果。
    */
    nn::Result AddRequestHeaderFormat(const char* format, ...) NN_NOEXCEPT;

    /*!
        @brief      投稿フィールドを追加します。

        @param[in]  data    データ。
        @param[in]  copy    コピーを作成するかどうか。

        @return     処理結果。
    */
    nn::Result SetPostField(const char* data, bool copy) NN_NOEXCEPT;

    /*!
        @brief      ヘッダの受信コールバックを登録します。

        @param[in]  function    ヘッダの受信コールバック。
        @param[in]  param       コールバックパラメータ。
    */
    void RegisterResponseHeaderCallback(ResponseHeaderCallback function, void* param) NN_NOEXCEPT;

    /*!
        @brief      ヘッダの受信完了コールバックを登録します。

        @param[in]  function    ヘッダの受信完了コールバック。
        @param[in]  param       コールバックパラメータ。

        @details
                    サーバから受信したヘッダをすべて受信した時点で本コールバックは呼び出されます。@n
                    また、ボディデータの受信前に呼び出されれることを保証します。
    */
    void RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback function, void* param) NN_NOEXCEPT;

    /*!
        @brief      データの送信コールバックを登録します。SetPostField の代わりに使用します。

        @param[in]  function    データの送信コールバック。
        @param[in]  param       コールバックパラメータ。

        @return     処理結果。
    */
    nn::Result RegisterSendDataCallback(HttpSendDataCallback function, void* param, size_t sendDataSize) NN_NOEXCEPT;

    /*!
        @brief      キャンセル可能オブジェクトを設定します。

        @param[in]  cancelable  キャンセル可能オブジェクト。

        @details
                    キャンセル可能オブジェクトを設定することで、入力処理を中断することが可能です。
    */
    void SetCancelable(const Cancelable* cancelable) NN_NOEXCEPT;

    //
    nn::Result Receive(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief  キャンセルされたかどうかをチェックし、されていれば
                コールバックでキャンセルしたことをマークして true を返します。
                (キャンセルされていなければ false を返します。)
    */
    bool CheckCancelStatusForCallback() NN_NOEXCEPT;

protected:
    /*!
        @brief      キャンセルされたかどうかを判定します。

        @return     キャンセルされたかどうか。

        @details
                    @ref SetCancelable でキャンセル可能オブジェクトを設定していた場合、 Cancelable::IsCanceled を呼び出します。@n
                    キャンセル可能オブジェクトが設定されていない場合、常に false を返します。
    */
    bool IsCanceled() const NN_NOEXCEPT;

private:
    /*!
        @brief      HTTP データ書き込み関数に渡すパラメータです。
    */
    struct HttpWriteFunctionParam
    {
        void* stream;                //!< 入力ストリームバッファ。
        size_t streamSize;           //!< 入力ストリームバッファのサイズ。
        size_t read;                 //!< 読み込んだサイズ。
        size_t positionInHttpBuffer; //!< HttpWriteFunction で取得できるバッファの読み込み位置。
        bool readNext;               //!< 次のバッファが読み込み可能かどうか。
    };

private:
    // Curlの型を出さない(curl.hをインクルードしない)ためにvoid*で定義
    void* m_Curl;
    void* m_Multi;
    void* m_Headers;
    //
    HttpWriteFunctionParam m_Param;
    //
    ResponseHeaderCallback m_ResponseHeaderCallback;
    void* m_ResponseHeaderParam;
    //
    ResponseHeaderEndCallback m_ResponseHeaderEndCallback;
    void* m_ResponseHeaderEndParam;
    //
    HttpSendDataCallback m_SendDataCallback;
    void* m_SendDataParam;
    size_t m_SendDataSize;
    //
    const Cancelable* m_Cancelable;
    //
    int m_StatusCode;
    //
    bool m_IsOpened;
    //
    bool m_IsCanceledInCallback;

private:
    //
    static size_t HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static size_t HttpReadFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static size_t HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
};

}}}}}
