﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

class FileStream;

/**
 * @brief ファイルシステムに対してファイルの読み書きを要求するクラスです。
 */
class FileSystem
{
public:
    /**
     * @brief 自プロセス向けにファイルシステムを利用できる状態にします。
     */
    static void InitializeFileSystem() NN_NOEXCEPT;
    /**
     * @brief ファイルシステムの利用を終了します。
     */
    static void FinalizeFileSystem() NN_NOEXCEPT;

    /**
     * @brief ファイルを読み取りモードで開きます。
     */
    static nn::Result OpenRead(FileStream* outStream, const char* fileName) NN_NOEXCEPT;

    /**
     * @brief ファイルを書き込みモード・追記書き込みモードで開きます。
     */
    static nn::Result OpenWrite(FileStream* outStream, const char* fileName) NN_NOEXCEPT;

    /**
     * @brief ファイルを書き込みモードで開きます。ファイルサイズを固定し、追記書き込みは行わないようにします。
     */
    static nn::Result OpenWrite(FileStream* outStream, const char* fileName, int64_t fileSize) NN_NOEXCEPT;

    /**
     * @brief ファイルを削除します。
     */
    static nn::Result Delete(const char* fileName) NN_NOEXCEPT;

    /**
     * @brief ディレクトリを作成します。
     */
    static nn::Result CreateDirectory(const char* directoryName) NN_NOEXCEPT;

    /**
     * @brief ディレクトリを削除します。
     */
    static nn::Result DeleteDirectory(const char* directoryName, bool recursive) NN_NOEXCEPT;

    /**
     * @brief 変更内容をコミットします。
     * @details
     * ファイルをオープン済みである場合、Commit 処理はブロックされます。
     */
    static nn::Result Commit() NN_NOEXCEPT;
};

/**
 * @brief オープンされたファイルへの読み書きを行うクラスです。
 */
class FileStream
{
public:
    FileStream() NN_NOEXCEPT :
        m_bOpened(false)
    {
    }
    /**
     * @brief デストラクターです。ファイルのクローズを行います。
     */
    ~FileStream() NN_NOEXCEPT
    {
        Close();
    }

    nn::Result Read(size_t* outReadSize, int64_t offset, void* outBuffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result Write(int64_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result Flush() NN_NOEXCEPT;
    nn::Result GetFileSize(int64_t* outFileSize) NN_NOEXCEPT;

private:
    void Close() NN_NOEXCEPT;

    nn::fs::FileHandle m_FileHandle;
    bool m_bOpened;

    friend class FileSystem;
};

}}}}}
