﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief PCIe Driver Public Data Types
 *
 * @details This file defines the public PCIe driver data types
 *          (C++ language).
 */

namespace nn {  namespace pcie {

/**
 * @brief Standard zero based bus identifier with valid range
 *        0-255 inclusive
 */
typedef uint32_t    BusNumber;

/**
 * @brief Standard zero based device identifier with valid range
 *        0-31 inclusive
 */
typedef uint8_t     DeviceNumber;

/**
 * @brief Standard zero based function identifier with valid
 *        range 0-7 inclusive
 */
typedef uint8_t     FunctionNumber;

/**
 * @brief Base class code.
 */
typedef uint8_t     BaseClassCode;

/**
 * @brief Sub class code.
 */
typedef uint8_t     SubClassCode;

/**
 * @brief Complete class code consisting of base, subclass and
 *        programming interface code.
 *          Bits  7-0  : programming interface code;
 *          Bits 15-8  : SubClassCode code;
 *          Bits 23-16 : BaseClassCode code.
 */
typedef uint32_t    FullClassCode;

/**
 * @brief Base + Sub class code;
 *          Bits 7-0  : SubClassCode code;
 *          Bits 15-8 : BaseClassCode code.
 */
typedef uint16_t     ClassCode;

/**
 * @brief Handle representing the runtime relationship
 *        established between the user's class driver and the
 *        core PCI stack. This handle tells the core PCI stack
 *        which user is making the API call.
 */
typedef uintptr_t   ClassDriverHandle;

const ClassDriverHandle InvalidClassDriverHandle = 0;
/**
 * @brief Handle representing the specific PCI device function
 *        which the user's class driver has acquired (taken
 *        ownership). This handle tells the core PCI stack to
 *        which function the invoked API applies.
 */
typedef uint32_t    FunctionHandle;

const FunctionHandle InvalidFunctionHandle = 0;

/**
 * @brief Physical address of data stored in RAM, accessible via
 *        PCI bus master. This is not a virtual address, and
 *        thus cannot be dereferenced by host CPU.
 */
typedef uint64_t    BusAddress;

/**
 * @brief Logical OR combination of any number of PCI region
 *        flags, as defined by the enumeration PciRegionFlag.
 *        Region flags describe behaviors of mapped PCI device
 *        function address space.
 */
typedef uint32_t    RegionFlagMask;

/**
 * @brief Logical OR combination of any number of PCI function
 *        select flags, as defined by the enumeration
 *        FunctionSelect. Class driver software
 *        uses these flags to indicate which fields within
 *        ClassDriverConfig are valid, refining the selection of
 *        endpoint devices which will be presented for probing.
 */
typedef uint32_t    FunctionSelectMask;

/**
 * @brief Logical OR combination of reset flags, as defined by the
 *        enumeration FunctionResetOption
 */
typedef uint32_t    FunctionResetOptionMask;

/**
 * @brief Speed of PCI bus to which a PCI device function is
 *        connected.
 */
enum BusSpeed
{
    BusSpeed_Invalid,
    BusSpeed_Auto,
    BusSpeed_EGen1,
    BusSpeed_EGen2,
    BusSpeed_EGen3
};

/**
 * @brief Power management state of a PCI device function.
 */
enum PmState
{
    PmState_D0 = 0,
    PmState_D1,
    PmState_D2,
    PmState_D3Hot,
    PmState_D3Off,
    PmState_Invalid,
};

/**
 * @brief Software interrupt service request mechanism used by
 *        a PCI device function.
 */
enum IrqType
{
    IrqType_Invalid = 0,
    IrqType_IntX,
    IrqType_Msi,
    IrqType_MsiX, //  not supported yet
};



/**
 * @brief   Specifies which fields of ClassDriverConfig are valid,
 *          to be considered as search criterion.
 */
enum FunctionSelect
{
    FunctionSelect_BaseClassCode = (1 << 0),
    FunctionSelect_SubClassCode  = (1 << 1),
    FunctionSelect_VendorId      = (1 << 2),
    FunctionSelect_DeviceId      = (1 << 3),
};

/**
 * @brief   Specifies a set of options which may be passed to ResetFunction().
 */
enum FunctionResetOption
{
    FunctionResetOption_None                  = 0,
    FunctionResetOption_DoExternalReset       = (1 << 0),
    FunctionResetOption_DoFunctionLevelReset  = (1 << 1),
    FunctionResetOption_DoNotReEnumerate      = (1 << 2),
};

/**
 * @brief Configuration specified by user's class driver upon
 *        registration.
 */
struct ClassDriverConfig
{
    uint64_t            classDriverContext;
    FunctionSelectMask  selectMask;
    BaseClassCode       baseClassCode;
    SubClassCode        subClassCode;
    uint16_t            vendorId;
    uint16_t            deviceId;
};

/**
 * @brief Profile of a specific device function
 */
struct FunctionState
{
    FunctionHandle functionHandle;
    BusNumber      busNum;
    DeviceNumber   deviceNum;
    FunctionNumber funcNum;
    uint16_t       vendorId;
    uint16_t       deviceId;
    uint8_t        headerType;
    FullClassCode  classCode;
    uint8_t        revision;
    BusSpeed       speed;
    int32_t        maxNumMsiVectors;
    PmState        pmState;
    bool           isAcquired;
};

const FunctionState InvalidFunctionState = {InvalidFunctionHandle, 255, 255, 255, 0, 0, 0, 0, 0,
                                            BusSpeed_Invalid, 0, PmState_Invalid, false};

/**
 * @brief Describe behaviors of mapped PCI device function
 *        address space.
 */
enum RegionFlag
{
    ResourceFlag_Io         = 0x0100,
    ResourceFlag_Mem        = 0x0200,
    ResourceFlag_Mem64      = 0x0400,
    ResourceFlag_Prefetch   = 0x0800,
    ResourceFlag_RawBarMask = 0x00ff,
};

/**
 * @brief Information about a PCI device function base address
 *        register (BAR)
 */
struct BarProfile
{
    size_t         size;
    RegionFlagMask flags;
};

/**
 * @brief Bus access size
 */
enum BusAccessWidth
{
    BusAccessWidth_8Bit,
    BusAccessWidth_16Bit,
    BusAccessWidth_32Bit,
    BusAccessWidth_64Bit,
};

/**
 * @brief Describes direct memory access (DMA) flow of data
 *        between PCI endpoint function and host RAM.
 */
enum DmaDirection
{
    DmaDirection_BiDirectional,
    DmaDirection_FromDevice,
    DmaDirection_ToDevice
};

/**
 * @brief Total maximum number of device functions which will be recognized
 */
const int32_t MaxEnumeratedDeviceFunctions = 4;

/**
 * @brief Total maximum number of device functions which may be acquired by a single class
 *        driver instantiation.
 */
const int32_t MaxDeviceFunctionsPerClient = 1;


/**
 * @brief Calls to MapDma() must ensure:
 *       - Specified base 'base' is aligned to
 *         MinimumDmaAddressAlignment.
 *       - Specified 'size' is a multiple of
 *         MinimumDmaAddressAlignment.
 */
const size_t MinimumDmaAddressAlignment = 4096;

/**
 * @brief Describes the maximum number of simultaneous users allowed by PCIe driver
 */
const int32_t MaxClientCount = 3;
const int32_t MaxLoggedStateClientCount = 2;

/**
 * @brief Maximum number of IRQs per device function
 */
const int32_t MaxIrqVectorsPerDeviceFunction = 32;

/**
 * @brief Describes the maximum number of simultaneous users allowed by PCIe driver
 */
const int32_t MaxSwitchPortCount = 2;





} // namespace pcie
} // namespace nn
