﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief PCIe Public API
 *
 * @details This file defines the public API (C++ language).
 */

namespace nn { namespace pcie {

/**
 * @brief Initialize PCIe API
 * @retresult
 *     @handleresult{ResultSuccess,           Initialization succeeded}
 * @endretresult
 * @pre
 *   - Initialize() has not yet been called, or
 *   - Finalize() has been previously called.
 * @post
 *   PCIe API can now be used.
 * @details
 *   This method prepares API for use, and thus must be called first.
 */
Result Initialize() NN_NOEXCEPT;

/**
 * @brief Finalize PCIe API
 * @retresult
 *     @handleresult{ResultSuccess,           Initialization succeeded}
 * @endretresult
 * @pre
 *   - Initialize() has been previously called.
 * @post
 *   - PCIe API can no longer be used.
 * @details
 *   This method releases internal resources, after which no other API may be called.
 */
Result Finalize() NN_NOEXCEPT;

/**
 * @brief Query functions without binding to class driver
 * @param[out] pOutFunctionState
 *   Returned array of PCIe device function state structures.
 * @param[out] pOutFunctionStateCount
 *   Returned value indicating number of PCIe device functions
 *   which have been returned via pOutFunctionState.
 * @param[in] functionStateBufferSize
 *   Size of user supplied memory referred to by pOutFunctionState.
 * @retresult
 *     @handleresult{ResultSuccess,           Query succeeded}
 * @endretresult
 * @pre
 *   Initialize() must be called once first.
 * @post
 *   This method only returns data and does not affect internal
 *   state of PCIe driver.
 * @details
 *   This method does not require association with a registered
 *   class driver, i.e. 'ClassDriverHandle'. The purpose of this method
 *   is to provide a way for an administrative entity to determine which
 *   device functions are currently enumerated.
 */
Result QueryFunctions(FunctionState*    pOutFunctionState,
                      int32_t*          pOutFunctionStateCount,
                      size_t            functionStateBufferSize) NN_NOEXCEPT;

/**
 * @brief Register user's class driver
 * @param[out]  pOutHandle
 *   User handle created by core PCIe stack.
 * @param[out] pOutEvent
 *   Supplied system event is initialized, and will provide indication
 *   if core PCIe stack detects any device functions matching the
 *   filter criteria supplied via ClassDriverConfig.
 * @param[in]  pInConfig
 *   User specific configuration table.
 * @param[in] eventClearMode
 *   Mode of operation assumed by initialized system event, related
 *   to how set event is cleared.
 * @retresult
 *     @handleresult{ResultSuccess,           Registration succeeded}
 *     @handleresult{ResultMemAllocFailure,   Internal memory allocation failured, cannot perform operation}
 *     @handleresult{ResultHandleAllocError,  Internal resources exhausted, cannot perform operation}
 * @endretresult
 * @pre
 *   Initialize() must be called once first.
 * @post
 *   Binding relationship now exists between core PCIe stack and
 *   the calling client.
 * @details
 *   Create binding relationship between user's class driver and core PCIe stack.
 */
Result RegisterClassDriver(ClassDriverHandle*       pOutHandle,
                           nn::os::SystemEventType* pOutEvent,
                           const ClassDriverConfig* pInConfig,
                           nn::os::EventClearMode   eventClearMode) NN_NOEXCEPT;

/**
 * @brief Unregister user's class driver
 * @param[in]  pInEvent
 *   System event which was passed by prior call to RegisterClassDriver().
 * @param[in]  classDriverHandle
 *   User's handle
 * @retresult
 *     @handleresult{ResultSuccess,                  Unregistration succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 * @endretresult
 * @pre
 *   - User has previously called RegisterClassDriver().
 *   - User has relinquished control over any previously acquired
 *     functions by calling ReleaseFunction().
 * @post
 *   Core PCIe stack no longer has record of this client.
 * @details
 *   Terminate binding relationship between user's class
 *   driver and core PCIe stack.
 */
Result UnregisterClassDriver(nn::os::SystemEventType* pInEvent,
                             ClassDriverHandle classDriverHandle) NN_NOEXCEPT;

/**
 * @brief Query functions
 * @param[out] pOutFunctionState
 *   Returned array of PCIe device function state structures.
 * @param[out] pOutFunctionStateCount
 *   Returned value indicating number of PCIe device functions
 *   which have been returned via pOutFunctionState.
 * @param[in]  classDriverHandle
 *   User's handle
 * @param[in] functionStateBufferSize
 *   Size of user supplied memory referred to by pOutFunctionState.
 * @retresult
 *     @handleresult{ResultSuccess,                  Query succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 * @endretresult
 * @pre
 *   User has active class driver registration, by having previously
 *   called RegisterClassDriver().
 * @post
 *   This method only returns data and does not affect internal
 *   state of PCIe driver.
 * @details
 *   The returned function list is limited by filtering parameters specified
 *   in ClassDriverConfig. Only functions which have not yet been acquired (available)
 *   are returned via this API. The whole point is to match a class driver with an available
 *   function.
 */
Result QueryFunctions(FunctionState*    pOutFunctionState,
                      int32_t*          pOutFunctionStateCount,
                      ClassDriverHandle classDriverHandle,
                      size_t            functionStateBufferSize) NN_NOEXCEPT;

/**
 * @brief Acquire function
 * @param[out] pOutEvent
 *   Supplied system event is initialized, and will provide indication
 *   if core PCIe stack detects any any power management state transitions
 *   for the specified endpoint device function.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of device endpoint function for which ownership is
 *   to be established.
 * @param[in] eventClearMode
 *   Mode of operation assumed by initialized system event, related
 *   to how set event is cleared.
 * @retresult
 *     @handleresult{ResultSuccess,                   Acquire succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultMaximumExceeded,           Single client acquired function limit, MaxDeviceFunctionsPerClient, exceeded.}
 *     @handleresult{ResultAlreadyAcquired,           Another class driver already acquired this function.}
 * @endretresult
 * @pre
 *   User has active class driver registration, by having previously
 *   called RegisterClassDriver().
 * @post
 *   - User now has exclusive ownership of the function.
 *   - User will be notified if function experiences power mode
 *     state changes.
 * @details
 *   Take exclusive ownership of an endpoint device function. This must be
 *   done prior to using any other API which accepts functionHandle as argument.
 */
Result AcquireFunction(nn::os::SystemEventType* pOutEvent,
                       ClassDriverHandle        classDriverHandle,
                       FunctionHandle           functionHandle,
                       nn::os::EventClearMode   eventClearMode) NN_NOEXCEPT;

/**
 * @brief Release function
 * @param[in]  pInEvent
 *   System event which was passed by prior call to AcquireFunction().
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of previously acquired endpoint function.
 * @retresult
 *     @handleresult{ResultSuccess,                   Release succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 * @endretresult
 * @pre
 *   - User has previously taken ownership of function by calling
 *     AcquireFunction().
 *   - Endpoint device can be in any state, including D3.
 * @post
 *   - Mappings created via prior calls to MapDma() are all
 *     automatically unmapped.
 *   - Previously applied IRQ settings are reverted.
 *   - User has relinquished control of function and is no longer
 *     able to interact with it in any way.
 * @details
 *   Relinquish ownership of an endpoint device function which was previously
 *   acquired via call to AcquireFunction().
 */
Result ReleaseFunction(nn::os::SystemEventType* pInEvent,
                       ClassDriverHandle        classDriverHandle,
                       FunctionHandle           functionHandle) NN_NOEXCEPT;

/**
 * @brief  Reset PCIe function
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  resetOptions
 *   Bit mask of selected options
 * @retresult
 *     @handleresult{ResultSuccess,                  Reset succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   - User has previously taken ownership of function by calling
 *     AcquireFunction().
 * @post
 *   Upon successful completion, class drivers should call ReleaseFunction(), as this functionHandle.
 *   will not be operational again.
 * @details
 *   This method will unconditionally end operational life of this functionHandle.
 *   This method will optionally reset the device, depending upon specified options.
 */
Result ResetFunction(ClassDriverHandle classDriverHandle,
                     FunctionHandle functionHandle,
                     FunctionResetOptionMask resetOptions) NN_NOEXCEPT;

/**
 * @brief Get function state
 * @param[out] pOutFunctionState
 *   Returned structure describing attributes and state of the
 *   specified device endpoint function.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of previously acquired endpoint function.
 * @retresult
 *     @handleresult{ResultSuccess,                   Retrieval succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 *   Endpoint device can be in any state, including D3.
 * @post
 *   This method only returns data and does not affect internal
 *   state of PCIe driver.
 * @details
 *   Current state of device function state is returned.
 */
Result GetFunctionState(FunctionState*    pOutFunctionState,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle) NN_NOEXCEPT;

/**
 * @brief Get BAR profile
 * @param[out] pOutBarProfile
 *   Returned structure containing information about the bar.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Zero based base address register number, valid range 0-5
 *   inclusive.
 * @retresult
 *     @handleresult{ResultSuccess,                   Retrieval succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   This method only returns data and does not affect internal
 *   state of PCIe driver.
 * @details
 *   Retrieves information about the specified base address register (BAR).
 */
Result GetBarProfile(BarProfile*       pOutBarProfile,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     uint8_t           bar) NN_NOEXCEPT;

/**
 * @brief Get virtual address mapping of BAR
 * @param[out] pOutMappedBar
 *   Returned pointer to BAR.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Zero based base address register number, valid range 0-5
 *   inclusive.
 * @retresult
 *     @handleresult{ResultSuccess,                   Retrieval succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   This method only returns data and does not affect internal
 *   state of PCIe driver.
 * @details
 *   Retrieves direct mapped pointer to specified base address register (BAR).
 *   THIS API WILL BE DEPRECATED IN NEAR FUTURE! CLASS DRIVERS SHOULD INSTEAD
 *   USE ReadBarRegion()/WriteBarRegion()!
 */
Result GetMappedBar(volatile uint8_t** pOutMappedBar,
                    ClassDriverHandle  classDriverHandle,
                    FunctionHandle     functionHandle,
                    uint8_t            bar) NN_NOEXCEPT;

/**
 * @brief Read 8-bit config value
 * @param[out] pOutData8
 *   Returned value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   Value read from config address space returned.
 * @details
 *   This method causes device config address space read cycle.
 */
Result ReadConfig8(uint8_t*          pOutData8,
                   ClassDriverHandle classDriverHandle,
                   FunctionHandle    functionHandle,
                   size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 16-bit config value
 * @param[out] pOutData16
 *   Returned value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                  Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,    Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   Value read from config address space returned.
 * @details
 *   This method causes device config address space read cycle.
 */
Result ReadConfig16(uint16_t*         pOutData16,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 32-bit config value
 * @param[out] pOutData32
 *   Returned value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   Value read from config address space returned.
 * @details
 *   This method causes device config address space read cycle.
 */
Result ReadConfig32(uint32_t*         pOutData32,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 8-bit config value
 * @param[in] data8
 *   Supplied value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to config address space.
 * @details
 *   This method causes device config address space write cycle.
 */
Result WriteConfig8(uint8_t           data8,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 16-bit config value
 * @param[in] data16
 *   Supplied value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to config address space.
 * @details
 *   This method causes device configuration address space write cycle.
 */
Result WriteConfig16(uint16_t          data16,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 32-bit config value
 * @param[in] data32
 *   Supplied value
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  offset
 *   Byte offset of configuration space to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to config address space.
 * @details
 *   This method causes device configuration address space write cycle.
 */
Result WriteConfig32(uint32_t          data32,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 8-bit value from BAR region
 * @param[out] pOutData8
 *   Returned data which was read from BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was read from BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O read cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result ReadBarRegion8(uint8_t*          pOutData8,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      uint8_t           bar,
                      size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 16-bit value from BAR region
 * @param[out] pOutData16
 *   Returned data which was read from BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was read from BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O read cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result ReadBarRegion16(uint16_t*         pOutData16,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 32-bit value from BAR region
 * @param[out] pOutData32
 *   Returned data which was read from BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was read from BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O read cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result ReadBarRegion32(uint32_t*         pOutData32,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read 64-bit value from BAR region
 * @param[out] pOutData64
 *   Returned data which was read from BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be read.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was read from BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O read cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result ReadBarRegion64(uint64_t*         pOutData64,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT;

/**
 * @brief Read BAR region
 * @param[out]  pOutData
 *   Returned data which was read from BAR region. Alignment
 *   of this pointer must correspond with specified accessWidth.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be read. Alignment
 *   must correspond with specified accessWidth.
 * @param[in]  accessWidth
 *   Width of read operation.
 * @param[in]  readSize
 *   Size of read operation in bytes.
 * @retresult
 *     @handleresult{ResultSuccess,                   Read succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultInvalidAccessWidth,        Specified access width is not valid}
 *     @handleresult{ResultInvalidRegisterSize,       Specified readSize is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Data was read from BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O read cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result ReadBarRegion(void *            pOutData,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     uint8_t           bar,
                     size_t            offset,
                     BusAccessWidth    accessWidth,
                     size_t            readSize) NN_NOEXCEPT;

/**
 * @brief Write 8-bit value to BAR region
 * @param[in] data8
 *   Supplied data which is to be written to BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O write cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result WriteBarRegion8(uint8_t           data8,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 16-bit value to BAR region
 * @param[in] data16
 *   Supplied data which is to be written to BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O write cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result WriteBarRegion16(uint16_t          data16,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 32-bit value to BAR region
 * @param[in] data32
 *   Supplied data which is to be written to BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O write cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result WriteBarRegion32(uint32_t          data32,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write 64-bit value to BAR region
 * @param[in] data64
 *   Supplied data which is to be written to BAR region.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be written.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Value was written to BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O write cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result WriteBarRegion64(uint64_t          data64,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT;

/**
 * @brief Write BAR region
 * @param[in]  pInData
 *   Supplied data which is written to BAR region. Alignment
 *   of this pointer must correspond with specified accessWidth.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  bar
 *   Base address register (BAR), zero based.
 * @param[in]  offset
 *   Byte offset of BAR region to be written. Alignment
 *   must correspond with specified accessWidth.
 * @param[in]  accessWidth
 *   Width of read operation.
 * @param[in]  writeSize
 *   Size of write operation in bytes.
 * @retresult
 *     @handleresult{ResultSuccess,                   Write succeeded}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultBarDoesNotExist,           Specified bar not valid for this device function}
 *     @handleresult{ResultInvalidRegisterOffset,     Specified offset is not valid}
 *     @handleresult{ResultInvalidAccessWidth,        Specified access width is not valid}
 *     @handleresult{ResultInvalidRegisterSize,       Specified writeSize is not valid}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Data was writen to BAR region.
 * @details
 *   This method causes device endpoint function MEM or I/O write cycles.
 *   Note that BAR regions are registers instantiated in the device endpoint
 *   and completely unrelated to host memory.
 */
Result WriteBarRegion(const void*       pInData,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      uint8_t           bar,
                      size_t            offset,
                      BusAccessWidth    accessWidth,
                      size_t            writeSize) NN_NOEXCEPT;

/**
 * @brief Find capability structure in device config space.
 * @param[out]  pOutOffset
 *   Returned offset within configuration space where the
 *   specified capability identifier is located.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  capability
 *   Capability identifier.
 * @retresult
 *     @handleresult{ResultSuccess,                   Capability structure located}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultNotAvailable,              Specified capability attribute not available}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   Offset of capability structure is returned.
 * @details
 *  - This refers to the original (first) PCI capability
 *    structure mechanism, where data is located within the
 *    first 255 bytes of configuration space.
 *  - See section 6.7 of "PCI Local Bus Specification 3.0" for
 *    details.
 *  - In header file pcie_Standard.h see definitions PCI_CAP_ID_*
 *    for relevant IDs.
 */
Result FindCapability(size_t*           pOutOffset,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      CapabilityId      capability) NN_NOEXCEPT;

/**
 * @brief Find extended capability structure in device config space.
 * @param[out]  pOutOffset
 *   Returned offset within configuration space where the
 *   specified capability identifier is located.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  extendedCapability
 *   Extended capability identifier.
 * @retresult
 *     @handleresult{ResultSuccess,                   Capability structure located}
 *     @handleresult{ResultInvalidClassDriverHandle,  Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,     Specified functionHandle is not valid}
 *     @handleresult{ResultNotAvailable,              Specified capability attribute not available}
 *     @handleresult{ResultFunctionIsInD3State,       Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction(), OR function has not yet been acquired by
 *   any class driver in the system.
 * @post
 *   Offset of capability structure is returned.
 * @details
 *  - This refers to the newer PCI capability
 *    structure mechanism introduced in PCI-X 2.0, where device
 *    functions have 4096 bytes of configuration space.
 *  - See section 6.7 of "PCI Local Bus Specification 3.0" for
 *    details.
 *  - In header file pcie_Standard.h see definitions
 *    PCI_EXT_CAP_ID_* for relevant IDs.
 */
Result FindExtendedCapability(size_t*              pOutOffset,
                              ClassDriverHandle    classDriverHandle,
                              FunctionHandle       functionHandle,
                              ExtendedCapabilityId extendedCapability) NN_NOEXCEPT;

/**
 * @brief   Map DMA
 * @param[out] pOutBusAddress
 *   Returned bus address corresponding corresponding with the base virtual address parameter.
 *   It is acceptable to pass NULL, if caller does not want to take receipt of buss address.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  direction
 *   Describes flow of data, specifying if mapped host memory is to be read or written by
 *   device endpoint function.
 * @param[in]  base
 *   Starting virtual address of RAM to be mapped. The memory allocation
 *   must be aligned and padded to nn::pcie::MinimumDmaAddressAlignment.
 * @param[in]  size
 *   Size of mapped RAM region. Must be a multiple of nn::pcie::MinimumDmaAddressAlignment.
 * @retresult
 *     @handleresult{ResultSuccess,                     Memory region was mapped}
 *     @handleresult{ResultInvalidClassDriverHandle,    Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,       Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidDmaDirection,         Specified direction is not valid}
 *     @handleresult{ResultInvalidAlignment,            Specified base is not aligned to MinimumDmaAddressAlignment}
 *     @handleresult{ResultInvalidSizePadding,          Specified size is not multiple of MinimumDmaAddressAlignment}
 *     @handleresult{ResultOverlappedDmaMapping,        At least part of the specified region has already been mapped}
 *     @handleresult{ResultFailedDmaMappingAllocation,  Internal resources exhausted, cannot perform operation}
 *     @handleresult{ResultFunctionIsInD3State,         Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Endpoint device function can now access the memory.
 * @details
 *   Make the specified region of host owned memory accessible to a bus mastering
 *   endpoint device function.
 */
Result MapDma(BusAddress*       pOutBusAddress,
              ClassDriverHandle classDriverHandle,
              FunctionHandle    functionHandle,
              DmaDirection      direction,
              void *            base,
              size_t            size) NN_NOEXCEPT;

/**
 * @brief   Unmap DMA
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  base
 *   Virtual address within region to be unmapped.
 *   If specified base is actually an address somewhere within the region, as opposed to the base,
 *   PCIe driver will locate the corresponding starting address and unmap the whole region.
 * @retresult
 *     @handleresult{ResultSuccess,                     Memory region was unmapped}
 *     @handleresult{ResultInvalidClassDriverHandle,    Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,       Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidAlignment,            Specified base is not aligned to MinimumDmaAddressAlignment}
 *     @handleresult{ResultNonExistentDmaMapping,       Specified mapped region does not exist}
 *     @handleresult{ResultFunctionIsInD3State,         Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has previously called MapDma() using the same base parameter.
 * @post
 *   Endpoint device function can no longer access the memory.
 * @details
 *   Make the specified region of host owned memory inaccessible to a bus mastering
 *   endpoint device function.
 */
Result UnmapDma(ClassDriverHandle classDriverHandle,
                FunctionHandle    functionHandle,
                void*             base) NN_NOEXCEPT;

/**
 * @brief   Unmap DMA
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  busAddress
 *   Bus address within region to be unmapped.
 *   If specified base is actually an address somewhere within the region, as opposed to the base,
 *   PCIe driver will locate the corresponding starting address and unmap the whole region.
 * @retresult
 *     @handleresult{ResultSuccess,                     Memory region was unmapped}
 *     @handleresult{ResultInvalidClassDriverHandle,    Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,       Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidAlignment,            Specified base is not aligned to MinimumDmaAddressAlignment}
 *     @handleresult{ResultNonExistentDmaMapping,       Specified mapped region does not exist}
 *     @handleresult{ResultFunctionIsInD3State,         Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User was given busAddress via prior call to MapDma().
 * @post
 *   Endpoint device function can no longer access the memory.
 * @details
 *   Make the specified region of host owned memory inaccessible to a bus mastering
 *   endpoint device function.
 */
Result UnmapDma(ClassDriverHandle classDriverHandle,
                FunctionHandle    functionHandle,
                BusAddress        busAddress)  NN_NOEXCEPT;

/**
 * @brief Get DMA bus address
 * @param[out] pOutBusAddress
 *   Returned bus address corresponding with supplied address.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  address
 *   Address of RAM for which bus address is to be retrieved.
 * @param[in] size
 *   Size of region that will be accessed. The only purpose
 *   of this parameter is for validating that a region has
 *   been mapped. One could always pass '1', if so desired.
 * @retresult
 *     @handleresult{ResultSuccess,                  Bus address was retrieved}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultNonExistentDmaMapping,    Region was not previously mapped}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User previously mapped memory region with call to MapDma().
 * @post
 *   Bus address is returned.
 * @details
 *   Specified address can be anywhere within region previously mapped
 *   by call to MapDma().
 */
Result GetDmaBusAddress(BusAddress*       pOutBusAddress,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        void*             address,
                        size_t            size) NN_NOEXCEPT;


/**
 * @brief Get DMA bus address range, base + size
 * @param[out] pOutBase
 *   Returned bus range base address
 * @param[out] pOutSize
 *   Returned bus range size
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @retresult
 *     @handleresult{ResultSuccess,                  Bus address range base and size were retrieved}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 * @post
 *   Bus address range base and size are returned.
 * @details
 *   Returns the DMA-able range.
 */
Result GetDmaBusAddressRange(BusAddress*       pOutBase,
                             BusAddress*       pOutSize,
                             ClassDriverHandle classDriverHandle,
                             FunctionHandle    functionHandle) NN_NOEXCEPT;

/**
 * @brief  Set DMA enable
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  isBusMasterEnabled
 *   True to allow device to master the bus.
 * @retresult
 *     @handleresult{ResultSuccess,                  Dma was enabled}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   Device is allowed to master the bus.
 * @details
 *   Enable or disable bus mastering capability of the
 *   specified device endpoint function. This is needed
 *   for:
 *   - DMA operations, where device writes to host owned
 *     memory, which has been mapped via MapDma().
 *   - Message Signalled Interrupt (MSI) operation,
 *     since it actually uses bus master DMA.
 */
Result SetDmaEnable(ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    bool              isBusMasterEnabled) NN_NOEXCEPT;

/**
 * @brief Acquire IRQ
 * @param[out] pOutEvent
 *   Supplied system event is initialized, and will provide indication
 *   when device endpoint IRQ activates.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in] eventClearMode
 *   Mode of operation assumed by initialized system event, related
 *   to how set event is cleared.
 * @param[in]  irqType
 *   Style of interrupt that user wishes to use for this
 *   particular device. When possible, MSI should be used
 *   instead of INTx, as it is more efficient internally.
 * @retresult
 *     @handleresult{ResultSuccess,                  Irq was acquired}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidArgument,          Specified irqType is not valid}
 *     @handleresult{ResultAlreadyAcquired,          Irq was already acquired previously}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has taken ownership of function by having called
 *   AcquireFunction().
 * @post
 *   System event may be signalled if device issues interrupt on bus.
 *   User has taken ownership of IRQ.
 * @details
 *   Take ownership of interrupt IRQ for this endpoint device function,
 *   allowing the class driver to implement an interrupt service routine.
 */
Result AcquireIrq(nn::os::SystemEventType* pOutEvent,
                  ClassDriverHandle        classDriverHandle,
                  FunctionHandle           functionHandle,
                  nn::os::EventClearMode   eventClearMode,
                  IrqType                  irqType) NN_NOEXCEPT;

/**
 * @brief  Release IRQ
 * @param[in]  pInEvent
 *   System event which was passed by prior call to AcquireIrq().
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 * @retresult
 *     @handleresult{ResultSuccess,                  Irq was released}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultIrqNotAcquired,           Irq was not previously acquired}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has previously acquired interrupt vector(s) by having called AcquireIrq().
 * @post
 *   User no longer has ownership of IRQ.
 * @details
 *   Relinquish ownership of previously acquired interrupt vector(s) for
 *   this endpoint device function, stopping further signalling of system event
 *   which was specified upon prior call to AcquireIrq().
 */
Result ReleaseIrq(nn::os::SystemEventType* pInEvent,
                  ClassDriverHandle        classDriverHandle,
                  FunctionHandle           functionHandle) NN_NOEXCEPT;

/**
 * @brief   Enable or disable the specified IRQ.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  zeroBasedIrqNumber
 *   Indicates which of the user's vectors for this particular
 *   endpoint device function should be enabled or disabled. For
 *   the typical case where only one vector was acquired, this
 *   value will be '0'.
 * @param[in]  isIrqEnabled
 *   If true, then specified interrupt vector is enabled.
 * @retresult
 *     @handleresult{ResultSuccess,                  Irq enable state was changed}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultInvalidArgument,          Specified zeroBasedIrqNumber is not valid}
 *     @handleresult{ResultIrqNotAcquired,           Irq was not previously acquired}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   User has previously acquired interrupt vector(s) by having
 *   called AcquireIrq().
 * @post
 *   State of Irq enable was changed.
 * @details
 *   For typical scenario where class driver uses only one Irq vector, zeroBasedIrqNumber will be zero.
 *   If irqType is IrqType_IntX, zeroBasedIrqNumber must always be zero.
 */
Result SetIrqEnable(ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    int32_t           zeroBasedIrqNumber,
                    bool              isIrqEnabled)  NN_NOEXCEPT;


/**
 * @brief  Enable or disable ASPM state for the function's device
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  isEnabled
 *   True to enable the ASPM state.  Otherwise, the ASPM state is disabled.
 * @retresult
 *     @handleresult{ResultSuccess,                  Dma was enabled}
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultHardwareUnsupported,      Specified ASPM state is not valid for port or device}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   Initialize() must be called once beforehand.
 * @post
 *   If supported, enables/disables ASPM low power states for the link.
 * @details
 *   Low power states enabled are L0s, L1, and L1SS.
 */
Result SetAspmEnable(ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     bool              isEnabled) NN_NOEXCEPT;

/**
 * @brief  Enable or disable policy which causes endpoint devices to be reset upon resume.
 * @param[in]  classDriverHandle
 *   User's handle.
 * @param[in]  functionHandle
 *   Handle of endpoint device function.
 * @param[in]  isEnabled
 *   'true' to request that endpoint device is reset upon system resume.
*    'false' to request that endpoint device state be preserved upon system resume.
 * @retresult
 *     @handleresult{ResultInvalidClassDriverHandle, Specified classDriverHandle is not valid}
 *     @handleresult{ResultInvalidFunctionHandle,    Specified functionHandle is not valid}
 *     @handleresult{ResultFunctionIsInD3State,      Function no longer actively enumerated}
 * @endretresult
 * @pre
 *   Initialize() must be called once beforehand.
 * @post
 *   Requested reset-upon-resume policy is applied next system resume.
 * @details
 *   Typically class drivers will want to set this policy to 'true', unless there
 *   is a compelling reason to maintain state across suspend-resume cycles.
 */
Result SetResetUponResumeEnable(ClassDriverHandle classDriverHandle,
                                FunctionHandle    functionHandle,
                                bool              isEnabled) NN_NOEXCEPT;


} // namespace pcie
} // namespace nn
