﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// olsc_MounterManager.h からのみ呼び出される前提

#include <algorithm>
#include <mutex>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TFormatString.h>

#if !defined(NN_SDK_BUILD_RELEASE)
#define NN_OLSC_MOUNT_MANAGER_LOG_ENABLE
#endif

#define NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(rtype, methodInfo) \
template <typename TagType, int MaxTagCount, int MaxMounterCount> \
inline rtype MountManager<TagType, MaxTagCount, MaxMounterCount>::methodInfo NN_NOEXCEPT

#define NN_OLSC_DEFINE_MOUNT_MANAGER_RET_VALUE(rtype) \
typename MountManager<TagType, MaxTagCount, MaxMounterCount>::rtype

#define NN_OLSC_DEFINE_MOUNT_MANAGER_CONST_RET_VALUE(rtype) \
const typename MountManager<TagType, MaxTagCount, MaxMounterCount>::rtype

namespace nn { namespace olsc { namespace srv { namespace util {

// ----------------------------------------------------------
// MountManager
// ----------------------------------------------------------

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(, MountManager())
    : m_RegisteredTagCount(0)
{
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, RegisterTag(TagType tag, const MountInfo& mountInfo))
{
    std::lock_guard<decltype(m_TagListLock)> lock(m_TagListLock);
    NN_SDK_ASSERT(m_RegisteredTagCount < MaxTagCount);

    m_RegisteredTags[m_RegisteredTagCount++] = { tag, mountInfo };
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(nn::util::optional<MounterId>, TryAcquire(TagType tag, const account::Uid& uid, ReferenceMode referenceMode))
{
    std::lock_guard<decltype(m_MounterLock)> lock(m_MounterLock);

    bool needsLog = false;

    auto proc = [&]() -> nn::util::optional<MounterId>
    {
        MounterHolder* overWritable = nullptr;
        MounterHolder* notUsed = nullptr;

        for (int i = 0; i < static_cast<int>(m_Mounters.size()); ++i)
        {
            auto& holder = m_Mounters[i];

            if (!holder)
            {
                notUsed = &holder;
            }
            else if (holder->GetTag() == tag && holder->GetUid() == uid)
            {
                if (holder->TryLock(referenceMode))
                {
                    return holder->GetId();
                }
                return nn::util::nullopt;
            }
            else
            {
                // Tag, Uid 不一致の Mounter で現在参照されていないものは
                // 上書き可能として記録する
                if (!holder->IsReferenced())
                {
                    overWritable = &holder;
                }
            }
        }

        // ここに来た時点で Mounter の再利用が不可能
        // 「上書き可能」は他から要求があった時に再利用できる可能性があるので
        // 「未使用」を優先して使用する。
        auto toEmplace = notUsed ? notUsed : overWritable;
        if (toEmplace)
        {
            (*toEmplace).emplace(uid, tag);
            auto& mounter = (*toEmplace).value();
            auto registeredTag = GetRegisterdTag(tag);

            NN_ABORT_UNLESS_RESULT_SUCCESS(mounter.EnsureAndMount(registeredTag.mountInfo));
            NN_ABORT_UNLESS(mounter.TryLock(referenceMode));

            needsLog = true; // 実際にマウントを行ったのでログを出す。

            return mounter.GetId();
        }

        return nn::util::nullopt;
    };

    auto result = proc();

    if (needsLog)
    {
        LogAcquire(result, tag, uid, referenceMode);
    }

    return result;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, Release(MounterId id, ReferenceMode referenceMode))
{
    std::lock_guard<decltype(m_MounterLock)> lock(m_MounterLock);

    auto mounter = FindMounterById(id);
    NN_SDK_ASSERT_NOT_NULL(mounter);
    mounter->Unlock(referenceMode);
    if (!mounter->IsReferenced())
    {
        m_AcquiableCondition.Signal();
    }
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(MounterId, Acquire(TagType tag, const account::Uid& uid, ReferenceMode referenceMode))
{
    std::lock_guard<decltype(m_MounterLock)> lock(m_MounterLock);
    nn::util::optional<MounterId> mounterId;
    while (!(mounterId = TryAcquire(tag, uid, referenceMode)))
    {
        LogTryAcquireFailed(tag, uid, referenceMode);
        m_AcquiableCondition.Wait(m_MounterLock);
    }

    return *mounterId;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(
    NN_OLSC_DEFINE_MOUNT_MANAGER_CONST_RET_VALUE(RegisteredTag&), GetRegisterdTag(TagType tag))
{
    for (int i = 0; i < m_RegisteredTagCount; ++i)
    {
        if (m_RegisteredTags[i].tag == tag)
        {
            return m_RegisteredTags[i];
        }
    }
    NN_ABORT("Not come here.\n");
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(
    NN_OLSC_DEFINE_MOUNT_MANAGER_RET_VALUE(Mounter*), FindMounterById(MounterId id))
{
    auto found = std::find_if(m_Mounters.begin(), m_Mounters.end(), [&](decltype(m_Mounters[0])& mounter) {
        if (!mounter)
        {
            return false;
        }
        return mounter->GetId() == id && mounter->IsReferenced();
    });
    return found == m_Mounters.end() ? nullptr : &(*found).value();
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(
    NN_OLSC_DEFINE_MOUNT_MANAGER_CONST_RET_VALUE(Mounter*), FindMounterById(MounterId id) const)
{
    const auto found = std::find_if(m_Mounters.begin(), m_Mounters.end(), [&](decltype(m_Mounters[0])& mounter) {
        if (!mounter)
        {
            return false;
        }
        return mounter->GetId() == id && mounter->IsReferenced();
    });
    return found == m_Mounters.end() ? nullptr : &(*found).value();
}


NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, LogCurrentState() const)
{
#if defined(NN_OLSC_MOUNT_MANAGER_LOG_ENABLE)
    NN_DETAIL_OLSC_TRACE("Current mount state:\n");
    for (int i = 0; i < static_cast<int>(m_Mounters.size()); ++i)
    {
        auto& mounter = m_Mounters[i];
        if (!mounter)
        {
            NN_DETAIL_OLSC_TRACE("[%d]: Empty\n", i);
        }
        else
        {
            const char* Format =
                "[%d]: Uid: %016llx%016llx\n"
                "     MounterId : %d\n"
                "     Tag       : %d\n"
                "     Referenced: %s\n"
                "     MountName : %s\n"
                "     Mode      : %s\n"
                "     RefCount  : %d\n";

            NN_DETAIL_OLSC_TRACE(Format,
                i, mounter->GetUid()._data[0], mounter->GetUid()._data[1],
                mounter->GetId(),
                mounter->GetTag(),
                mounter->IsReferenced() ? "True" : "False",
                mounter->GetMountName(),
                (mounter->GetReferenceMode() == ReferenceMode::Read ? "Read" : "Write"),
                mounter->GetReferenceCount());
        }
    }
#endif
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, LogAcquire(nn::util::optional<MounterId> result, TagType tag, const account::Uid& uid, ReferenceMode referenceMode) const)
{
#if defined(NN_OLSC_MOUNT_MANAGER_LOG_ENABLE)
    NN_DETAIL_OLSC_TRACE("Acquire mounter: Tag: %d, Uid: %016llx%016llx, Mode: %s\n", tag, uid._data[0], uid._data[1],
        referenceMode == ReferenceMode::Read ? "Read" : "Write");
    if (result)
    {
        NN_DETAIL_OLSC_TRACE("Acquire result : Success (MounterId = %d)\n", *result);
    }
    else
    {
        NN_DETAIL_OLSC_TRACE("Acquire result : Failure\n");
    }
    LogCurrentState();
#else
    NN_UNUSED(result);
    NN_UNUSED(tag);
    NN_UNUSED(uid);
    NN_UNUSED(referenceMode);
#endif
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, LogTryAcquireFailed(TagType tag, const account::Uid& uid, ReferenceMode referenceMode) const)
{
#if defined(NN_OLSC_MOUNT_MANAGER_LOG_ENABLE)
    NN_DETAIL_OLSC_TRACE("TryAcquire failed. Tag: %d, Uid : %016llx%016llx, Mode : %s\n", tag, uid._data[0], uid._data[1],
        referenceMode == ReferenceMode::Read ? "Read" : "Write");
    NN_DETAIL_OLSC_TRACE("Waiting for mounter slot to be available.");
    LogCurrentState();
#else
    NN_UNUSED(tag);
    NN_UNUSED(uid);
    NN_UNUSED(referenceMode);
#endif
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(const char*, GetMountName(MounterId id) const)
{
    std::lock_guard<decltype(m_MounterLock)> lock(m_MounterLock);

    auto mounter = FindMounterById(id);
    NN_SDK_ASSERT_NOT_NULL(mounter);
    return mounter->GetMountName();
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(const char*, GetRootPath(MounterId id) const)
{
    std::lock_guard<decltype(m_MounterLock)> lock(m_MounterLock);

    auto mounter = FindMounterById(id);
    NN_SDK_ASSERT_NOT_NULL(mounter);
    return mounter->GetRootPath();
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(ReadMount, AcquireForRead(TagType tag, const account::Uid& uid))
{
    auto mounter = Acquire(tag, uid, ReferenceMode::Read);
    return GetMountContext<ReadMount>(mounter);
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(WriteMount, AcquireForWrite(TagType tag, const account::Uid& uid))
{
    auto mounter = Acquire(tag, uid, ReferenceMode::ReadAndWrite);
    return GetMountContext<WriteMount>(mounter);
}

// ----------------------------------------------------------
// MountManager::Mounter
// ----------------------------------------------------------

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(, Mounter::Mounter(const account::Uid& uid, TagType tag))
    :   m_Uid(uid), m_Tag(tag), m_RefCount(0)
{
    InitializeId();
}


NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, Mounter::InitializeId())
{
    NN_FUNCTION_LOCAL_STATIC(MounterId, s_Id = 0);
    m_Id = s_Id;
    s_Id = (s_Id + 1) % 1000;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(, Mounter::~Mounter())
{
    Unmount();
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(Result, Mounter::EnsureAndMount(const MountInfo& mountInfo))
{
    NN_RESULT_DO(Ensure(mountInfo));
    NN_RESULT_DO(Mount(mountInfo));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(Result, Mounter::Ensure(const MountInfo& mountInfo))
{
    if (m_Uid != account::InvalidUid)
    {
        NN_RESULT_TRY(fs::CreateSystemSaveData(mountInfo.systemSaveDataId, m_Uid, mountInfo.saveDataSize, mountInfo.journalSize, mountInfo.saveDataFlags))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        NN_RESULT_END_TRY;
    }
    else
    {
        NN_RESULT_TRY(fs::CreateSystemSaveData(mountInfo.systemSaveDataId, mountInfo.saveDataSize, mountInfo.journalSize, mountInfo.saveDataFlags))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        NN_RESULT_END_TRY;
    }
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(Result, Mounter::Mount(const MountInfo& mountInfo))
{
    NN_ABORT_UNLESS(nn::util::TSNPrintf(m_MountName, sizeof(m_MountName), "olsc_%03d", m_Id) < sizeof(m_MountName));
    NN_ABORT_UNLESS(nn::util::TSNPrintf(m_RootPath, sizeof(m_RootPath), "%s:/", m_MountName) < sizeof(m_RootPath));

    if (m_Uid != account::InvalidUid)
    {
        NN_RESULT_DO(fs::MountSystemSaveData(m_MountName, mountInfo.systemSaveDataId, m_Uid));
    }
    else
    {
        NN_RESULT_DO(fs::MountSystemSaveData(m_MountName, mountInfo.systemSaveDataId));
    }

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, Mounter::Unmount())
{
    fs::Unmount(m_MountName);
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(TagType, Mounter::GetTag() const)
{
    return m_Tag;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(const account::Uid&, Mounter::GetUid() const)
{
    return m_Uid;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(bool, Mounter::IsReferenced() const)
{
    return m_RefCount > 0;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(const char*, Mounter::GetMountName() const)
{
    return m_MountName;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(const char*, Mounter::GetRootPath() const)
{
    return m_RootPath;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(MounterId, Mounter::GetId() const)
{
    return m_Id;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(int, Mounter::GetReferenceCount() const)
{
    return m_RefCount;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(ReferenceMode, Mounter::GetReferenceMode() const)
{
    return m_ReferenceMode;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(bool, Mounter::TryLock(ReferenceMode referenceMode))
{
    bool result;
    if (referenceMode == ReferenceMode::Read)
    {
        result = m_ReaderWriterLock.TryAcquireReadLock();
    }
    else if (referenceMode == ReferenceMode::ReadAndWrite)
    {
        result = m_ReaderWriterLock.TryAcquireWriteLock();
    }
    else
    {
        NN_ABORT("Not come here.\n");
    }

    if (result)
    {
        if (m_RefCount == 0)
        {
            m_ReferenceMode = referenceMode;
        }
        m_RefCount++;
    }

    return result;
}

NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD(void, Mounter::Unlock(ReferenceMode referenceMode))
{
    if (referenceMode == ReferenceMode::Read)
    {
        m_ReaderWriterLock.ReleaseReadLock();
    }
    else if (referenceMode == ReferenceMode::ReadAndWrite)
    {
        m_ReaderWriterLock.ReleaseWriteLock();
    }
    else
    {
        NN_ABORT("Not come here.\n");
    }

    NN_SDK_ASSERT(m_RefCount > 0);
    m_RefCount--;
}

}}}} // nn::olsc::srv::util

#undef NN_OLSC_DEFINE_MOUNT_MANAGER_METHOD
#undef NN_OLSC_DEFINE_MOUNT_MANAGER_RET_VALUE
#undef NN_OLSC_DEFINE_MOUNT_MANAGER_CONST_RET_VALUE
