﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  サーバーに対して発行する API です。
 */

#include <nn/fs/fs_SaveDataTransferForCloudBackUp.h>
#include <nn/util/util_Execution.h>

#include <curl/curl.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

    /**
    *       @brief      URL を含んだ ComponentFile の情報を取得します。
    *
    *       @param[out]  pOut   サーバーから返ってくる ComponentFileInfo。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   id   ComponentFileInfo の ID。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result GetComponentFileSignedUrlForDownload(ComponentFileInfo* pOut, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ダウンロード用 S3 バケットに対して、落としたい ComponentFile をダウンロードします。
    *                   pOutSize で得られた Size と ComponentFileInfo にかかれている saveDataChunkSize が一致していることを確認して下さい。
    *
    *       @param[out]  pOutSize   ComponentFile のサイズ。
    *       @param[out]  outBuffer   ComponentFile を格納するバッファ。
    *       @param[in]   bufferSize   outBuffer のサイズ。
    *       @param[in]   url   ComponentFile の URL。
    *       @param[in]   downloadSize    ダウンロードする ComponentFile のサイズ。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestDownloadComponentFile(size_t* pOutSize, void* outBuffer, size_t bufferSize, const char* url, size_t downloadSize, CURL* curlHandle, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ダウンロード用 S3 バケットに対して、指定した ComponentFile の URL からデータを取得後し importer。 に渡します。
    *
    *       @param[in]   importer   ダウンロードするデータの importer。
    *       @param[in]   url   ComponentFile の URL。
    *       @param[in]   importSize    インポートする ComponentFile のサイズ。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestImportComponentFile(fs::ISaveDataChunkImporter* importer, const char* url, size_t importSize, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

}}}} // namespace nn::olsc::srv::transfer
