﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/olsc_AutoUploadSettingManager.h>
#include <nn/olsc/srv/olsc_SaveDataArchiveInfoCacheManager.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/util/olsc_ActiveUserList.h>
#include <nn/olsc/srv/util/olsc_AutonomyUploadWaitingList.h>
#include <nn/olsc/srv/util/olsc_Os.h>
#include <nn/olsc/srv/util/olsc_SaveDataAccessLockManager.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv {

class UserActivityHandler
{
    NN_DISALLOW_COPY(UserActivityHandler);
    NN_DISALLOW_MOVE(UserActivityHandler);

public:
    UserActivityHandler(
        util::ActiveUserList& activeUserList,
        util::SaveDataAccessLockManager& saveDataAccessLockManager,
        database::TransferTaskDatabaseManager& ttdm,
        SaveDataArchiveInfoCacheManager& sdaInfoCacheManager,
        SeriesInfoDatabaseManager& seriesInfoDatabaseManager,
        TransferTaskAgent& agent) NN_NOEXCEPT
        :   m_ActiveUserList(activeUserList),
            m_SaveDataAccessLockManager(saveDataAccessLockManager),
            m_Ttdm(ttdm),
            m_SdaInfoCacheManager(sdaInfoCacheManager),
            m_SeriesInfoDatabaseManager(seriesInfoDatabaseManager),
            m_Agent(agent)
    {}

    // NSA ユーザに変化があった時に呼ばれる
    void RefreshActiveUserList() NN_NOEXCEPT;

    // アプリ開始・終了のタイミングで呼ばれる
    void EnableSaveDataAccessLock(ApplicationId appId) NN_NOEXCEPT;
    void AppendSaveDataAccessLock(ApplicationId appId, const ApplicationId ids[], int count) NN_NOEXCEPT;
    void DisableSaveDataAccessLock(ApplicationId appId) NN_NOEXCEPT;

    // 自動 UL 待ちリストに登録されているアプリのセーブデータのタスク登録を行う
    // (アプリ終了時・スリープ遷移時に呼び出される)
    void ProcessAutonomyUploadWaitingList(nn::util::Cancelable& cancelable) NN_NOEXCEPT;
    nn::os::SystemEvent& GetAutonomyUploadWaitingListUpdateEvent() NN_NOEXCEPT;

    // 自動実行待ち間隔を終えているアクティブユーザのセーブデータに対してタスク登録・SDA の更新を行う
    // (定期自動 UL チェックで呼び出される)
    void ExecuteUserTaskIfScheduled(nn::util::Cancelable& cancelable) NN_NOEXCEPT;
    nn::os::SystemEvent&    GetUserTaskScheduleEvent() NN_NOEXCEPT;
    TimeSpan    GetNextUserTaskExecutionTime() NN_NOEXCEPT;

    // 実行中アプリのセーブアップロードの許可／不許可の設定
    // (全起床時や、全起床→スリープなどのタイミングで呼ばれる想定)
    void SuppressSaveDataAccessLockForUpload() NN_NOEXCEPT;
    void UnsuppressSaveDataAccessLockForUpload() NN_NOEXCEPT;

private:
    util::ActiveUserList& m_ActiveUserList;
    util::AutonomyUploadWaitingList m_AutonomyUploadWaitingList;
    util::SaveDataAccessLockManager& m_SaveDataAccessLockManager;

    database::TransferTaskDatabaseManager& m_Ttdm;
    SaveDataArchiveInfoCacheManager& m_SdaInfoCacheManager;
    SeriesInfoDatabaseManager& m_SeriesInfoDatabaseManager;
    TransferTaskAgent& m_Agent;

    os::SdkRecursiveMutex m_Lock;
    nn::util::optional<ApplicationId> m_CurrentRunningApplication = nn::util::nullopt;

    void RegisterAutonomyUploadTask(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT;
};

}}} // namespace nn::olsc::srv
