﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_Result.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/olsc_SaveDataArchiveInfoCacheManager.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskFactory.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_Os.h>
#include <nn/olsc/srv/util/olsc_ThreadResource.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>

namespace nn { namespace olsc { namespace srv {

class TransferTaskAgent
{
    NN_DISALLOW_COPY(TransferTaskAgent);
    NN_DISALLOW_MOVE(TransferTaskAgent);
public:
    using TransferTaskDatabaseManager = database::TransferTaskDatabaseManager;
    using ThreadResource = util::ThreadResource<24 * 1024>;

    TransferTaskAgent(
        TransferTaskDatabaseManager& ttdm,
        ThreadResource& threadResource,
        TransferTaskFactoryBase& taskFactory,
        TransferTaskStartEventManager& taskStartEventManager,
        TransferTaskCompleteEventManager& taskCompleteEventManager,
        SeriesInfoDatabaseManager& seriesInfoDatabaseManager,
        SaveDataArchiveInfoCacheManager& sdaInfoCacheManager,
        database::ErrorHistoryDatabase& errorHistoryDatabase,
        TransferTaskBase::TransferTaskExecutionResource& executionResource) NN_NOEXCEPT;
    ~TransferTaskAgent() NN_NOEXCEPT;

    void CancelCurrentTask() NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

    void IncrementStopExecutionCount() NN_NOEXCEPT;
    void DecrementStopExecutionCount() NN_NOEXCEPT;

    class TaskLocker
    {
    public:
        NN_IMPLICIT TaskLocker(os::SdkRecursiveMutex& lock) NN_NOEXCEPT
            : m_Lock(lock)
        {
            m_Lock.Lock();
        }
        ~TaskLocker()
        {
            m_Lock.Unlock();
        }
    private:
        os::SdkRecursiveMutex& m_Lock;
    };

    TaskLocker  LockTask() NN_NOEXCEPT;
    TransferTaskBase* GetCurrentTask() NN_NOEXCEPT;

private:
    void Execute() NN_NOEXCEPT;

    enum class AgentState : Bit8
    {
        Terminated = 0,
        TaskExecutable = 1,
        TaskNonExecutable = 2,
    };

    AgentState PerformExecutableState() NN_NOEXCEPT;
    AgentState PerformNonExecutableState() NN_NOEXCEPT;

    bool ProcessTaskList() NN_NOEXCEPT;
    bool CanExecute() const NN_NOEXCEPT;
    void StartRetryWaiting() NN_NOEXCEPT;
    void ProcessPostProcessEvent() NN_NOEXCEPT;
    void ClearCurrentTask() NN_NOEXCEPT;
    void ResetCurlHandle() NN_NOEXCEPT;

    nn::util::optional<TransferTaskDetailInfo> GetExecutableTaskDetailInfo() NN_NOEXCEPT;

    TransferTaskDatabaseManager& m_Ttdm;
    ThreadResource& m_ThreadResource;
    os::SdkRecursiveMutex m_ExecutionLock;
    mutable os::SdkRecursiveMutex m_TaskLock;
    bool m_Started;

    TransferTaskBase* m_CurrentTask;
    TransferTaskFactoryBase& m_TaskFactory;
    os::ThreadType m_TaskExecutionThread;

    int m_ExecutionStopCount;

    TransferTaskStartEventManager& m_TaskStartEventManager;
    TransferTaskCompleteEventManager& m_TaskCompleteEventManager;

    TransferTaskBase::TransferTaskExecutionResource& m_ExecutionResource;

    bool m_RetryWaiting;
    SeriesInfoDatabaseManager& m_SeriesInfoDatabaseManager;
    SaveDataArchiveInfoCacheManager& m_SdaInfoCacheManager;
    database::ErrorHistoryDatabase& m_ErrorHistoryDatabase;

    util::ManualClearEvent m_TerminateEvent;
    util::ManualClearEvent m_PostProcessEvent;
    util::ManualClearEvent m_ExecutionStopEvent;
    util::ManualClearEvent m_TaskCreatedEvent;
    util::ManualClearEvent m_TaskRetryEvent;
    util::ManualClearTimerEvent m_RetryWaitingEvent;
    util::ManualClearTimerEvent m_CurlTimerEvent;

    // TODO : context 統合後削除
    TransferTaskSuspendedInfo m_SuspendedInfo = {};
};

}}} // ~namespace nn::olsc::srv
