﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <curl/curl.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_Network.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace olsc { namespace srv {

class TransferTaskBase : public nn::util::Cancelable
{
    NN_DISALLOW_COPY(TransferTaskBase);

public:
    static const int MaxDivisionCount = 65;

    using TransferTaskContext = database::TransferTaskContextDatabase::TransferTaskContext;
    using SaveDataInfoListBuffer = std::array<SaveDataArchiveInfo, MaxApplicationCount>;
    // TODO : url のサイズが 512 Byte に戻ったら 512 * 1024 に戻す
    using WorkBuffer = std::array<char, 1024 * 1024>;
    using ComponentFileInfoListBuffer = std::array<ComponentFileInfo, MaxDivisionCount>;

    struct TransferTaskExecutionResource
    {
        WorkBuffer workBuffer;
        NsaIdToken nsaIdTokenBuffer;
        CURL* curlHandle;
    };


    TransferTaskBase(
        TransferTaskExecutionResource& executionResource) NN_NOEXCEPT
        :   m_ExecutionResource(executionResource),
            m_Connection(os::EventClearMode_ManualClear)
    {}

    virtual ~TransferTaskBase() NN_NOEXCEPT
    {}

    virtual void Execute() NN_NOEXCEPT = 0;
    bool IsSucceeded() const NN_NOEXCEPT;
    bool IsRetryable() const NN_NOEXCEPT;
    bool IsFatal() const NN_NOEXCEPT;
    void Resume(const TransferTaskDetailInfo& detailInfo, const TransferTaskContext& context, const SeriesInfo& lastSi) NN_NOEXCEPT;
    // TODO : suspendedInfo を context に統合後削除
    void Resume(const TransferTaskDetailInfo& detailInfo, const TransferTaskContext& context, const SeriesInfo& lastSi, TransferTaskSuspendedInfo&& info) NN_NOEXCEPT;
    void Initialize(const TransferTaskDetailInfo& detailInfo, const SeriesInfo& lastSi) NN_NOEXCEPT;
    const TransferTaskContext& GetContext() const NN_NOEXCEPT;
    const nn::util::optional<SaveDataArchiveInfo>& TryGetLatestSaveDataArchive() const NN_NOEXCEPT;
    const TransferTaskDetailInfo& GetDetailInfo() const NN_NOEXCEPT;
    // TODO : suspendedInfo を context に統合後削除
    TransferTaskSuspendedInfo DetachTransferTaskSuspendedInfo() NN_NOEXCEPT;

protected:
    void CompleteTask(Result result) NN_NOEXCEPT;

    void InitializeContext(TransferTaskId id) NN_NOEXCEPT;
    void SetupContext(size_t dataSize, uint32_t maxPartitionsCount) NN_NOEXCEPT;

    using ContextUpdater = std::function<void(TransferTaskContext& context)>;
    void UpdateContext(const ContextUpdater& pred) NN_NOEXCEPT;
    void UpdateContextByChunk(size_t chunkSize) NN_NOEXCEPT;

    // TODO : ちゃんとした Context の取得ができたら排除
    void CompleteContext() NN_NOEXCEPT;

    TransferTaskExecutionResource& GetExecutionResource() NN_NOEXCEPT;
    void PrintDetailInfo() const NN_NOEXCEPT;
    const SeriesInfo& GetLastSi() const NN_NOEXCEPT;
    void SetLatestSaveDataArchive(const nn::util::optional<SaveDataArchiveInfo>& sda) NN_NOEXCEPT;
    Result CheckAndGetServerSda(nn::util::optional<SaveDataArchiveInfo>* pOutSda, SaveDataArchiveInfo serverSdaList[], int listCount, bool isForce) NN_NOEXCEPT;
    nifm::NetworkConnection& GetNetworkConnection() NN_NOEXCEPT;
    void UpdateTransferTaskSuspendedInfo(TransferTaskSuspendedInfo&& suspendedInfo) NN_NOEXCEPT;
    bool IsTaskResumable() NN_NOEXCEPT;

private:
    Result m_LastResult = ResultSuccess();
    bool m_Done = false;
    mutable os::SdkRecursiveMutex m_Lock;
    TransferTaskContext m_Context {};

    TransferTaskDetailInfo m_DetailInfo {};
    TransferTaskExecutionResource& m_ExecutionResource;
    SeriesInfo m_LastSi;
    nn::util::optional<SaveDataArchiveInfo> m_LatestSaveDataArchive;
    nifm::NetworkConnection m_Connection;
    // TODO : suspendedInfo を context に統合後削除
    TransferTaskSuspendedInfo m_SuspendedInfo {};

    virtual bool IsRetryableResult(Result result) const NN_NOEXCEPT = 0;
};

class UploadTransferTask : public TransferTaskBase
{
    NN_DISALLOW_COPY(UploadTransferTask);
public:
    NN_IMPLICIT UploadTransferTask(TransferTaskExecutionResource& executionResource) NN_NOEXCEPT
        : TransferTaskBase(executionResource)
    {}

    virtual void Execute() NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual bool IsRetryableResult(Result result) const NN_NOEXCEPT NN_OVERRIDE;
    Result Upload() NN_NOEXCEPT;
    Result PrepareNewUpload(
        bool* pOutIsDiffUpload,
        SaveDataArchiveInfo* pOutSda,
        std::unique_ptr<fs::ISaveDataDivisionExporter>* pOutExporter,
        const TransferTaskDetailInfo& detailInfo,
        fs::SaveDataTransferManagerForCloudBackUp& manager,
        TransferTaskExecutionResource& resource
    ) NN_NOEXCEPT;

    Result PrepareResumedUpload(
        SaveDataArchiveInfo* pOutSda,
        const TransferTaskDetailInfo& detailInfo,
        SaveDataArchiveId sdaId,
        fs::ISaveDataChunkIterator& iter,
        TransferTaskExecutionResource& resource
    ) NN_NOEXCEPT;

};

class DownloadTransferTask : public TransferTaskBase
{
    NN_DISALLOW_COPY(DownloadTransferTask);
public:
    NN_IMPLICIT DownloadTransferTask(TransferTaskExecutionResource& executionResource) NN_NOEXCEPT
        : TransferTaskBase(executionResource)
    {}

    virtual void Execute() NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual bool IsRetryableResult(Result result) const NN_NOEXCEPT NN_OVERRIDE;

    Result Download() NN_NOEXCEPT;
};


}}} // ~namespace nn::olsc::srv
