﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/database/olsc_PolicyInfoCache.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/olsc_AutoUploadSettingManager.h>
#include <nn/olsc/srv/olsc_DaemonControllerImpl.h>
#include <nn/olsc/srv/olsc_Executor.h>
#include <nn/olsc/srv/olsc_TransferTaskFactory.h>
#include <nn/olsc/srv/olsc_TransferTaskListControllerImpl.h>
#include <nn/olsc/srv/olsc_RemoteStorageControllerImpl.h>
#include <nn/olsc/srv/olsc_SaveDataArchiveInfoCacheManager.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/olsc_Service.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_LmemUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv {

class ServiceResource
{
public:

    static const util::MountInfo DeviceSaveInfo;
    static const util::MountInfo UserSettingSaveInfo;
    static const util::MountInfo UserSeriesInfoSaveInfo;

    ServiceResource() NN_NOEXCEPT
        : m_MountManager(DeviceSaveInfo, UserSettingSaveInfo, UserSeriesInfoSaveInfo)
        , m_SeriesInfoDatabaseManager(m_MountManager)
        , m_SdaInfoCacheManager(m_MountManager)
        , m_AutoUploadSettingManager(m_MountManager)
        , m_TransferTaskDatabaseManager(m_MountManager)
        , m_ErrorHistoryDatabase(m_MountManager)
        , m_PolicyInfoCache(m_MountManager)
        , m_TransferTaskListController(
            m_ObjectMemoryResourceHolder.GetMemoryResourceRef(),
            m_TransferTaskDatabaseManager,
            m_TransferTaskAgent,
            m_TransferTaskStartEventManager,
            m_TransferTaskCompleteEventManager)
        , m_RemoteStorageController(
            m_ObjectMemoryResourceHolder.GetMemoryResourceRef(),
            m_TransferTaskDatabaseManager,
            m_SeriesInfoDatabaseManager,
            m_SdaInfoCacheManager,
            m_Executor,
            m_MountManager)
        , m_DaemonController(
            m_ObjectMemoryResourceHolder.GetMemoryResourceRef(),
            m_AutoUploadSettingManager,
            m_TransferTaskDatabaseManager)
        , m_TransferTaskFactory(m_TransferTaskDatabaseManager)
        , m_TtaThreadResource(NN_SYSTEM_THREAD_NAME(olsc, TransferTask), NN_SYSTEM_THREAD_PRIORITY(olsc, TransferTask))
        , m_TransferTaskAgent(
            m_TransferTaskDatabaseManager,
            m_TtaThreadResource,
            m_TransferTaskFactory,
            m_TransferTaskStartEventManager,
            m_TransferTaskCompleteEventManager,
            m_SeriesInfoDatabaseManager,
            m_SdaInfoCacheManager,
            m_ErrorHistoryDatabase,
            m_TransferTaskExecutionResource)
        , m_Service(
            m_ObjectMemoryResourceHolder.GetMemoryResourceRef(),
            m_RemoteStorageController,
            m_TransferTaskListController,
            m_DaemonController,
            m_SeriesInfoDatabaseManager,
            m_SdaInfoCacheManager,
            m_ErrorHistoryDatabase,
            m_PolicyInfoCache,
            m_TransferTaskAgent,
            m_Executor)
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        m_TransferTaskAgent.Start();
    }

    sf::SharedPointer<IOlscServiceForSystemService> GetServiceForSystemService() NN_NOEXCEPT;

    Executor& GetExecutor() NN_NOEXCEPT
    {
        return m_Executor;
    }

private:
    // 非同期オブジェクト生成用のバッファ
    template <size_t BufferSize>
    class MemoryResourceHolder
    {
    private:
        NN_STATIC_ASSERT(BufferSize % sizeof(Bit64) == 0);

        Bit64 m_Buffer[BufferSize / sizeof(Bit64)];
        lmem::HeapHandle m_Handle;
        nn::util::optional<sf::ExpHeapMemoryResource> m_pMemoryResource;
    public:
        MemoryResourceHolder() NN_NOEXCEPT;
        ~MemoryResourceHolder() NN_NOEXCEPT;
        MemoryResource& GetMemoryResourceRef() NN_NOEXCEPT;
    };
    // TODO サイズ調整
    MemoryResourceHolder<8 * 1024> m_ObjectMemoryResourceHolder;

    util::DefaultMountManager m_MountManager;
    SeriesInfoDatabaseManager m_SeriesInfoDatabaseManager;
    SaveDataArchiveInfoCacheManager m_SdaInfoCacheManager;
    TransferTaskCompleteEventManager m_TransferTaskCompleteEventManager;
    TransferTaskStartEventManager m_TransferTaskStartEventManager;
    AutoUploadSettingManager m_AutoUploadSettingManager;

    database::TransferTaskDatabaseManager m_TransferTaskDatabaseManager;
    database::ErrorHistoryDatabase m_ErrorHistoryDatabase;
    database::PolicyInfoCache m_PolicyInfoCache;

    TransferTaskListControllerImpl m_TransferTaskListController;
    RemoteStorageControllerImpl m_RemoteStorageController;
    DaemonControllerImpl m_DaemonController;

    TransferTaskFactory m_TransferTaskFactory;
    TransferTaskAgent::ThreadResource m_TtaThreadResource;
    TransferTaskAgent m_TransferTaskAgent;

    TransferTaskBase::TransferTaskExecutionResource m_TransferTaskExecutionResource;
    Executor m_Executor;
    Service m_Service;
};

}}} // ~namespace nn::olsc::srv

// 実装

namespace nn { namespace olsc { namespace srv {

template <size_t BufferSize>
ServiceResource::MemoryResourceHolder<BufferSize>::MemoryResourceHolder() NN_NOEXCEPT
{
    m_Handle = lmem::CreateExpHeap(m_Buffer, sizeof(m_Buffer), lmem::CreationOption_ThreadSafe);
    NN_ABORT_UNLESS_NOT_NULL(m_Handle);
    m_pMemoryResource.emplace(m_Handle);
}
template <size_t BufferSize>
ServiceResource::MemoryResourceHolder<BufferSize>::~MemoryResourceHolder() NN_NOEXCEPT
{
    m_pMemoryResource = nn::util::nullopt;
    lmem::DestroyExpHeap(m_Handle);
}
template <size_t BufferSize>
MemoryResource& ServiceResource::MemoryResourceHolder<BufferSize>::GetMemoryResourceRef() NN_NOEXCEPT
{
    return *m_pMemoryResource;
}

}}} // ~namespace nn::olsc::srv
