﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Result.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/olsc/srv/database/olsc_ExecutionList.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/olsc_UserReusableObjectManager.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class TransferTaskDatabase
{
    NN_DISALLOW_COPY(TransferTaskDatabase);

public:
    static const int ExecutionListCacheCount = 1024;
    static const int ExecutionListReadBufferCount = 256;
    static const int DetailInfoStoreReadBufferCount = 128;

public:
    NN_IMPLICIT TransferTaskDatabase(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        :   m_ExecutionInfoList(mountManager),
            m_Metadata({}), m_ExecutionListUpdatedEvent(os::EventClearMode_ManualClear, false), m_MountManager(mountManager), m_DetailInfoStoreManager(mountManager)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(Initialize());
    }
    virtual ~TransferTaskDatabase() NN_NOEXCEPT
    {}

    Result AppendTask(TransferTaskId* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config, TransferTaskRank rank) NN_NOEXCEPT;

    Result RemoveTask(TransferTaskId id) NN_NOEXCEPT;
    Result MoveTaskToHead(TransferTaskId id) NN_NOEXCEPT;

    int ListTasks(TransferTaskDetailInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT;
    Result GetTask(TransferTaskDetailInfo* out, TransferTaskId id) const NN_NOEXCEPT;
    Result GetExecutionInfoByIndex(ExecutionList::ExecutionInfo* out, int index) const NN_NOEXCEPT;

    using FindTaskPredicate = std::function<bool(const TransferTaskDetailInfo& detailInfo)>;
    nn::util::optional<TransferTaskDetailInfo> FindTaskIf(const FindTaskPredicate& pred) const NN_NOEXCEPT;
    int GetTaskCount() const NN_NOEXCEPT;
    Result CleanupUserData(const account::Uid& uid) NN_NOEXCEPT;
    Result RemoveTaskIfMatch(const account::Uid& uid, const std::function<bool(const TransferTaskDetailInfo& detailInfo)>& pred) NN_NOEXCEPT;

    os::SystemEvent& GetExecutionListUpdatedEvent() NN_NOEXCEPT;
    Result UpdateTaskState(TransferTaskId id, TransferTaskDetailInfo::State state) NN_NOEXCEPT;

    Result RemoveAllTask() NN_NOEXCEPT;

    void ForEachDetailInfo(const account::Uid& uid, const std::function<bool(const TransferTaskDetailInfo& detailInfo)>& pred) NN_NOEXCEPT;

private:
    struct DetailInfoEntry
    {
        TransferTaskId          id;
        account::Uid            uid;
        ApplicationId           appId;
        TransferTaskConfig      config;
        TransferTaskDetailInfo::State state;

        inline bool operator==(const DetailInfoEntry& rhs) const NN_NOEXCEPT
        {
            return this->id == rhs.id
                && this->uid == rhs.uid
                && this->appId == rhs.appId
                && this->config == rhs.config
                && this->state == rhs.state;
        }
        inline bool operator!=(const DetailInfoEntry& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }
    };

    using DetailInfoStoreBase = DataArray<util::DefaultMountManager, DetailInfoEntry, MaxApplicationCount, 0, DetailInfoStoreReadBufferCount>;
    class DetailInfoStore : public DetailInfoStoreBase
    {
    public:
        DetailInfoStore(const account::Uid& uid, ReadBuffer* readBuffer, util::DefaultMountManager& mountManager) : DetailInfoStoreBase(&m_CacheBuffer, readBuffer, mountManager)
        {
            NN_ABORT_UNLESS(nn::util::TSNPrintf(m_MetadataPath, sizeof(m_MetadataPath), "ttdb_td_%016llx%016llx_meta", uid._data[0], uid._data[1]) < sizeof(m_MetadataPath));
            NN_ABORT_UNLESS(nn::util::TSNPrintf(m_EntryPath, sizeof(m_EntryPath), "ttdb_td_%016llx%016llx_entry", uid._data[0], uid._data[1]) < sizeof(m_EntryPath));

            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::ReadMount AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::WriteMount AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE;
    private:
        char m_MetadataPath[64];
        char m_EntryPath[64];
        CacheBuffer m_CacheBuffer;
    };

    class DetailInfoStoreManager : public UserReusableObjectManager<DetailInfoStore, 1>
    {
    public:
        NN_IMPLICIT DetailInfoStoreManager(util::DefaultMountManager& mountManager) : m_MountManager(mountManager)
        {}

        util::WriteMount AcquireWriteMount() NN_NOEXCEPT
        {
            return m_MountManager.AcquireDeviceSaveForWrite();
        }

        util::ReadMount AcquireReadMount() NN_NOEXCEPT
        {
            return m_MountManager.AcquireDeviceSaveForRead();
        }
    protected:
        virtual void EmplaceObject(nn::util::optional<DetailInfoStore>& toEmplace, const account::Uid& uid) NN_NOEXCEPT NN_OVERRIDE
        {
            toEmplace.emplace(uid, &m_ReadBuffer, m_MountManager);
        }

    private:
        util::DefaultMountManager& m_MountManager;
        DetailInfoStore::ReadBuffer m_ReadBuffer;
    };

    struct Metadata
    {
        int currentId;
    };

    ExecutionList m_ExecutionInfoList;
    mutable os::SdkRecursiveMutex m_Lock;
    Metadata m_Metadata;

    os::SystemEvent m_ExecutionListUpdatedEvent;
    util::DefaultMountManager& m_MountManager;
    mutable DetailInfoStoreManager m_DetailInfoStoreManager;

    const char* MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) NN_NOEXCEPT;

    Result WriteMetadataFile() NN_NOEXCEPT;
    Result ReadMetadataFile() NN_NOEXCEPT;
    Result CreateDetailInfoEntry(DetailInfoEntry* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config) NN_NOEXCEPT;
    Result AddDetailInfoEntry(const DetailInfoEntry& detailInfo) NN_NOEXCEPT;
    Result RemoveDetailInfoEntry(const account::Uid& uid, TransferTaskId id) NN_NOEXCEPT;
    nn::util::optional<DetailInfoEntry> FindDetailInfoEntry(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT;
    nn::util::optional<DetailInfoEntry> FindDetailInfoEntry(const account::Uid& uid, TransferTaskId id) const NN_NOEXCEPT;
    nn::util::optional<ExecutionList::ExecutionInfo> FindExecutionInfo(TransferTaskId id) const NN_NOEXCEPT;
    Result GenerateId(TransferTaskId* out) NN_NOEXCEPT;
    void SignalExecutionListUpdatedEvent() NN_NOEXCEPT;

    Result Initialize() NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database

