﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Result.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class TransferTaskContextDatabase
{
    NN_DISALLOW_COPY(TransferTaskContextDatabase);

public:
    static const int MaxTransferTaskContextCount = 8;

    struct TransferTaskContext
    {
        TransferTaskId      id;
        size_t              totalSize;
        size_t              currentSize;
        size_t              througput;
        int                 totalComponentCount;
        int                 currentComponentIndex;
        ComponentFileInfo   currentComponentFile;
        Result              lastResult;
    };

    static const int ReadBufferCount = MaxTransferTaskContextCount;
    using Database = DataArray<util::DefaultMountManager, TransferTaskContext, MaxTransferTaskContextCount, 0, ReadBufferCount>;
    class TransferTaskContextStore : public Database
    {
    public:
        NN_IMPLICIT TransferTaskContextStore(util::DefaultMountManager& mountManager) NN_NOEXCEPT
            : Database(&m_Cache, &m_ReadBuffer, mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::ReadMount AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::WriteMount AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE;
    private:
        CacheBuffer m_Cache;
        ReadBuffer m_ReadBuffer;
    };

public:
    NN_IMPLICIT TransferTaskContextDatabase(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        : m_Store(mountManager)
    {}
    virtual ~TransferTaskContextDatabase() NN_NOEXCEPT
    {}
    Result GetTransferTaskContext(TransferTaskContext* out, TransferTaskId id) const NN_NOEXCEPT;

    nn::util::optional<TransferTaskContext> GetTransferTaskContext(TransferTaskId id) const NN_NOEXCEPT;

    Result RemoveTransferTaskContext(TransferTaskId id) NN_NOEXCEPT;
    Result CreateTransferTaskContext(TransferTaskId id, int totalComponentCount, size_t totalSize) NN_NOEXCEPT;
    Result UpdateTransferTaskContext(const TransferTaskContext& context) NN_NOEXCEPT;

    // デバッグ用
    Result RemoveAllTransferTaskContext() NN_NOEXCEPT;
private:
    TransferTaskContextStore            m_Store;
    mutable os::SdkRecursiveMutex       m_Lock;

private:
    void CreateTransferTaskContext(TransferTaskContext* out, TransferTaskId id, int componentFileCount, size_t totalFileSize) NN_NOEXCEPT;
};

inline bool operator==(const TransferTaskContextDatabase::TransferTaskContext& lhs, const TransferTaskContextDatabase::TransferTaskContext& rhs) NN_NOEXCEPT
{
    return lhs.id == rhs.id
        && lhs.totalSize == rhs.totalSize
        && lhs.currentSize == rhs.currentSize
        && lhs.througput == rhs.througput
        && lhs.totalComponentCount == rhs.totalComponentCount
        && lhs.currentComponentIndex == rhs.currentComponentIndex
        && lhs.currentComponentFile == rhs.currentComponentFile
        && lhs.lastResult.GetInnerValueForDebug() == rhs.lastResult.GetInnerValueForDebug();
}

inline bool operator!=(const TransferTaskContextDatabase::TransferTaskContext& lhs, const TransferTaskContextDatabase::TransferTaskContext& rhs) NN_NOEXCEPT
{
    return !(lhs.id == rhs.id);
}

}}}} // namespace nn::olsc::srv::database

