﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  SeriesInfo のデータベース。
 */

#include <nn/nn_ApplicationId.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class SeriesInfoDatabase
{
    NN_DISALLOW_COPY(SeriesInfoDatabase);

public:
    struct Entry
    {
        ApplicationId appId;
        SeriesInfo seriesInfo;
    };

    NN_IMPLICIT SeriesInfoDatabase(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
        : m_Db(uid, mountManager)
    {
    }

    ~SeriesInfoDatabase() NN_NOEXCEPT
    {}

    nn::util::optional<SeriesInfo> Find(ApplicationId appId) const NN_NOEXCEPT;
    SeriesInfo Get(ApplicationId) const NN_NOEXCEPT;
    Result Update(ApplicationId appId, const SeriesInfo& seriesInfo) NN_NOEXCEPT;
    Result Remove(ApplicationId appId) NN_NOEXCEPT;
    Result Clear() NN_NOEXCEPT;


    void lock() NN_NOEXCEPT
    {
        m_Db.lock();
    }

    void unlock() NN_NOEXCEPT
    {
        m_Db.unlock();
    }
private:
    static const int MaxCacheCount = 0;
    static const int ReadBufferCount = 256;
    using DatabaseBase = DataArray<util::DefaultMountManager, Entry, MaxApplicationCount, MaxCacheCount, ReadBufferCount>;
    class DatabaseImpl : public DatabaseBase
    {
    public:
        DatabaseImpl(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
            : DatabaseBase(&m_CacheBuffer, &m_ReadBuffer, mountManager), m_Uid(uid)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::ReadMount AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::WriteMount AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE;

    private:
        const account::Uid m_Uid;
        CacheBuffer m_CacheBuffer;
        ReadBuffer m_ReadBuffer;

    };

    mutable DatabaseImpl m_Db;
};

inline bool operator==(const SeriesInfoDatabase::Entry& lhs, const SeriesInfoDatabase::Entry& rhs) NN_NOEXCEPT
{
    return lhs.appId == rhs.appId
        && lhs.seriesInfo == rhs.seriesInfo;
}

inline bool operator!=(const SeriesInfoDatabase::Entry& lhs, const SeriesInfoDatabase::Entry& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

}}}} // namespace nn::olsc::srv::database
