﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  SaveDataArchiveInfoCache の操作を行うインターフェースです。
 */

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/olsc_Types.h>
#include <nn/olsc/srv/database/olsc_CachedReaderWriter.h>
#include <nn/olsc/srv/database/olsc_Datastore.h>
#include <nn/olsc/srv/database/olsc_SortedDataArray.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class SaveDataArchiveInfoCache
{
    NN_DISALLOW_COPY(SaveDataArchiveInfoCache);

public:
    SaveDataArchiveInfoCache(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
        : m_Store(uid, mountManager), m_AppIdList(uid, mountManager), m_IdList(uid, mountManager)
    {
    }

    ~SaveDataArchiveInfoCache() NN_NOEXCEPT
    {}

    int GetCount() const NN_NOEXCEPT;

    nn::util::optional<DataInfo> GetDataInfoByApplicationId(ApplicationId id) const NN_NOEXCEPT;
    nn::util::optional<DataInfo> GetDataInfoByDataId(DataId id) const NN_NOEXCEPT;
    nn::util::optional<SaveDataArchiveInfo> GetSaveDataArchiveInfoByApplicationId(ApplicationId id) const NN_NOEXCEPT;
    nn::util::optional<SaveDataArchiveInfo> GetSaveDataArchiveInfoByDataId(DataId id) const NN_NOEXCEPT;

    Result Add(const SaveDataArchiveInfo& sdaInfo) NN_NOEXCEPT;

    Result Delete(ApplicationId id) NN_NOEXCEPT;
    Result Delete(DataId id) NN_NOEXCEPT;
    Result DeleteAll() NN_NOEXCEPT;

    // 実際に取得した Sda の数が返り値
    int List(SaveDataArchiveInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT;
    int ListDataInfo(DataInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT;

    void lock() NN_NOEXCEPT
    {
        m_AppIdList.lock();
        m_IdList.lock();
        m_Store.lock();
    }
    void unlock() NN_NOEXCEPT
    {
        m_Store.unlock();
        m_IdList.unlock();
        m_AppIdList.unlock();
    }

private:
    static const int MaxEntryCount = MaxApplicationCount;

    class SaveDataArchiveInfoStore : public database::Datastore<util::DefaultMountManager, SaveDataArchiveInfo, MaxEntryCount>
    {
    public:
        SaveDataArchiveInfoStore(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
            : database::Datastore<util::DefaultMountManager, SaveDataArchiveInfo, MaxEntryCount>(mountManager), m_Uid(uid)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::ReadMount AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::WriteMount AcquireWriteMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
    private:
        account::Uid m_Uid;
    };

    SaveDataArchiveInfoStore m_Store;

    static const int SaveDataArchiveInfoKeyListReadBufferCount = MaxEntryCount / 8;


    class SaveDataArchiveInfoAppIdList : public SortedDataArray<ApplicationId, int, SaveDataArchiveInfoKeyListReadBufferCount>
    {
    public:
        SaveDataArchiveInfoAppIdList(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT :
            SortedDataArray<ApplicationId, int, SaveDataArchiveInfoKeyListReadBufferCount>(Compare, &m_ReadBuffer, MaxEntryCount), m_MountManager(mountManager), m_Uid(uid)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sda_app_id_list_meta";
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sda_app_id_list_entry";
        }

        virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForRead(m_Uid);
        }

        virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForWrite(m_Uid);
        }

    private:
        static int Compare(const ApplicationId& lhs, const ApplicationId& rhs) NN_NOEXCEPT
        {
            if (lhs.value == rhs.value)
            {
                return 0;
            }
            else if (lhs.value < rhs.value)
            {
                return -1;
            }
            return 1;
        }
        ReadBuffer m_ReadBuffer;
        util::DefaultMountManager& m_MountManager;
        account::Uid m_Uid;
    };

    class SaveDataArchiveInfoIdList : public SortedDataArray<SaveDataArchiveId, int, SaveDataArchiveInfoKeyListReadBufferCount>
    {
    public:
        SaveDataArchiveInfoIdList(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT :
            SortedDataArray<SaveDataArchiveId, int, SaveDataArchiveInfoKeyListReadBufferCount>(Compare, &m_ReadBuffer, MaxEntryCount), m_MountManager(mountManager), m_Uid(uid)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sda_id_list_meta";
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sda_id_list_entry";
        }

        virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForRead(m_Uid);
        }

        virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForWrite(m_Uid);
        }

    private:
        static int Compare(const SaveDataArchiveId& lhs, const SaveDataArchiveId& rhs) NN_NOEXCEPT
        {
            if (lhs == rhs)
            {
                return 0;
            }
            else if (lhs < rhs)
            {
                return -1;
            }
            return 1;
        }
        ReadBuffer m_ReadBuffer;
        util::DefaultMountManager& m_MountManager;
        account::Uid m_Uid;
    };

    SaveDataArchiveInfoAppIdList m_AppIdList;
    SaveDataArchiveInfoIdList m_IdList;

    template <typename T>
    int ListImpl(T out[], int maxOutCount, int offset, T (*convert)(SaveDataArchiveInfo&& sda) ) const NN_NOEXCEPT;

    template<typename IndexListType, typename IdType>
    Result DeleteImpl(IndexListType& indexList, IdType id) NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database


