﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  自動アップロードのデータベース設定
 */

#include <nn/nn_ApplicationId.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_SortedDataArray.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class AutoUploadSettingDatabase
{
    NN_DISALLOW_COPY(AutoUploadSettingDatabase);

public:
    AutoUploadSettingDatabase(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
        : m_SettingList(uid, mountManager), m_Uid(uid), m_MountManager(mountManager)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadGlobalSetting());
    }

    void SetGlobalSetting(bool isEnabled) NN_NOEXCEPT;
    bool GetGlobalSetting() const NN_NOEXCEPT;

    enum class ApplicationAutoUploadSetting : Bit8
    {
        NotSet      = 0,    // 未設定
        Enabled     = 1,    // 自動 UL 有効
        Disabled    = 2,    // 自動 UL 無効
    };

    void SetApplicationAutoUploadSetting(ApplicationId appId, ApplicationAutoUploadSetting setting) NN_NOEXCEPT;
    ApplicationAutoUploadSetting GetApplicationAutoUploadSetting(ApplicationId appId) const NN_NOEXCEPT;
    void RemoveApplicationAutoUploadSetting(ApplicationId appId) NN_NOEXCEPT;
    void Cleanup() NN_NOEXCEPT;

    using ForEachPredicate = std::function<void(ApplicationId appId, ApplicationAutoUploadSetting setting)>;
    void ForEach(ForEachPredicate pred) NN_NOEXCEPT;

    void lock() NN_NOEXCEPT
    {
        m_GeneralSettingLock.lock();
        m_SettingList.lock();
    }
    void unlock() NN_NOEXCEPT
    {
        m_SettingList.unlock();
        m_GeneralSettingLock.unlock();
    }
private:
    static const int ReadBufferCount = 64;
    static const int MaxEntryCount = MaxApplicationCount;
    struct ApplicationSetting
    {
        ApplicationAutoUploadSetting setting;
    };

    class SettingList : public database::SortedDataArray<ApplicationId, ApplicationSetting, ReadBufferCount>
    {
    public:
        SettingList(const account::Uid& uid, util::DefaultMountManager& mountManager) NN_NOEXCEPT
            :   database::SortedDataArray<ApplicationId, ApplicationSetting, ReadBufferCount>(Compare, &m_ReadBuffer, MaxEntryCount),
                m_MountManager(mountManager), m_Uid(uid)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "auto_ul_setting_meta";
        }
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "auto_ul_setting_entry";
        }
        virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForRead(m_Uid);
        }
        virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireUserSettingsSaveForWrite(m_Uid);
        }

    private:
        static int Compare(const ApplicationId& lhs, const ApplicationId& rhs) NN_NOEXCEPT
        {
            if (lhs.value == rhs.value)
            {
                return 0;
            }
            else if (lhs.value < rhs.value)
            {
                return -1;
            }
            return 1;
        }

        ReadBuffer m_ReadBuffer;
        util::DefaultMountManager& m_MountManager;
        const account::Uid m_Uid;
    };

    SettingList m_SettingList;

    static const int MaxPathLen = 64;

    mutable os::SdkRecursiveMutex m_GeneralSettingLock;

    struct GlobalSetting
    {
        bool isEnabled;
    };

    GlobalSetting m_GlobalSetting {};
    const account::Uid m_Uid;
    util::DefaultMountManager& m_MountManager;

    const char* MakeGlobalSettingFilePath(char* out, int maxOutCount, const char* rootPath) const NN_NOEXCEPT;
    Result SetGlobalSettingImpl(bool isEnabled) NN_NOEXCEPT;
    Result GetGlobalSettingImpl() NN_NOEXCEPT;
    Result ReadGlobalSetting() NN_NOEXCEPT;
    void SetApplicationSettingsAll(ApplicationAutoUploadSetting setting) NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database
