﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  オンラインストレージの操作を行うためのインターフェースです。
 */

#include <nn/nn_ApplicationId.h>
#include <nn/account/account_Types.h>
#include <nn/olsc/olsc_AsyncRequest.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/olsc_Types.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace olsc {
    namespace srv {
        class IRemoteStorageController;
    }

/**
*   @brief  リモートストレージに対する操作を行うオブジェクトです。
*/
class RemoteStorageController {
    NN_DISALLOW_COPY(RemoteStorageController);

public:

    RemoteStorageController() NN_NOEXCEPT;
    ~RemoteStorageController() NN_NOEXCEPT;

    RemoteStorageController& operator=(RemoteStorageController&& rhs) NN_NOEXCEPT;
    RemoteStorageController(RemoteStorageController&& rhs) NN_NOEXCEPT;

    /**
    *   @brief      データ情報キャッシュの更新をリクエストします。
    *
    *   @param[out] outAsync          非同期オブジェクトの格納先です。
    *
    *   @return     処理結果を返します。
    *
    *   @pre
    *       -       インターネット接続要求が受理されている。
    *
    *   @details    データ情報キャッシュの更新をリクエストします。
    *               データ情報キャッシュが更新される際、各アプリケーションに対するデータポリシーキャッシュも同時に更新されます。
    *
    *               この関数で同時にリクエストできる非同期リクエスト数は 1 でこのクラスの他のリクエストと共有しています。
    *
    */
    Result RequestUpdateDataInfoCacheAsync(AsyncRequest* outAsync) NN_NOEXCEPT;

    /**
    *   @brief      アプリケーションを指定してデータ情報キャッシュを更新することをリクエストします。
    *
    *   @param[out] outAsync            非同期オブジェクトの格納先です。
    *   @param[in]  id                  対象となるアプリケーション。
    *
    *   @return     処理結果を返します。
    *
    *   @pre
    *       -       インターネット接続要求が受理されている。
    *
    *   @details    指定した id に一致するデータ情報キャッシュの更新をリクエストします。
    *               データ情報キャッシュが更新される際、id の一致するデータポリシーキャッシュも同時に更新されます。
    *
    *               この関数で同時にリクエストできる非同期リクエスト数は 1 でこのクラスの他のリクエストと共有しています。
    *
    */
    Result RequestUpdateDataInfoCacheAsync(AsyncRequest* outAsync, ApplicationId id) NN_NOEXCEPT;

    /**
    *   @brief      データ情報キャッシュを消去します。
    *
    *   @return     処理結果を返します。
    *
    *   @details    データ情報キャッシュの更新を消去します。
    *               本 API 実行後、データ情報キャッシュを使用する場合は RequestUpdateDataInfoCache または RequestUpdateDataInfoCacheAsync を実行してください。
    *
    */
    Result ClearDataInfoCache() NN_NOEXCEPT;

    /**
    *   @brief      データ情報を指定してそれに紐づくサムネイルのキャッシュを更新します。
    *
    *   @param[out] outAsync            非同期オブジェクトの格納先です。
    *   @param[in]  id                  対象のデータ。
    *
    *   @return     処理結果を返します。
    *
    *   @pre
    *       -       インターネット接続要求が受理されている。
    *
    *   @details    id に指定したデータ情報に紐づくサムネイルのキャッシュの更新をリクエストします。
    *               id のデータに対してすでにサムネイルがキャッシュされている場合は、outAsync は直ちに完了します。
    *               この場合は通信を行いません。
    *
    *               この関数で同時にリクエストできる非同期リクエスト数は 1 でこのクラスの他のリクエストと共有しています。
    *
    */
    Result RequestUpdateThumbnailCache(AsyncRequest* outAsync, DataId id) NN_NOEXCEPT;

    /**
    *   @brief      リモートストレージ上のデータの削除をリクエストします。
    *
    *   @param[out] outAsync    非同期オブジェクトの格納先です。
    *   @param[in]  id          対象のデータ。
    *
    *   @return     処理結果を返します。
    *
    *   @pre
    *       -       インターネット接続要求が受理されている。
    *
    *   @details    id に指定したデータの削除をリクエストします。関連するキャッシュも同時に削除されます。
    *               リクエスト実行時にサーバ上のデータの状態が Persisted 以外の場合は失敗します。
    *
    *               この関数で同時にリクエストできる非同期リクエスト数は 1 でこのクラスの他のリクエストと共有しています。
    *
    */
    Result RequestDeleteData(AsyncRequest* outAsync, DataId id) NN_NOEXCEPT;


    /**
    *   @brief      セーブデータをリモートストレージ上のデータとしてアップロードする転送タスクを登録します。
    *
    *   @param[out] out         作成された転送タスクの ID。
    *   @param[in]  id          対象のアプリケーション。
    *
    *   @pre
    *           - outValue != nullptr
    *
    *   @return     処理結果を返します。
    *
    *   @details    id で指定したセーブデータをリモートストレージ上のデータとしてアップロードする転送タスクを登録します。
    *               TransferTaskListController の各関数に outValue の値を渡すことで、進捗を確認したり完了を待つことができます。
    *
    */
    Result RegisterUploadSaveDataTransferTask(TransferTaskId* out, ApplicationId id) NN_NOEXCEPT;

    /**
    *   @brief      リモートストレージ上のデータをセーブデータとしてダウンロードする転送タスクを登録します。
    *
    *   @param[out] out         作成された転送タスクの ID。
    *   @param[in]  id          対象のデータ。
    *
    *   @pre
    *           - outValue != nullptr
    *
    *   @return     処理結果を返します。
    *
    *   @details    id で指定したリモートストレージ上のデータを、セーブデータとしてダウンロードする転送タスクを登録します。
    *               TransferTaskListController の各関数に outValue の値を渡すことで、進捗を確認したり完了を待つことができます。
    *
    */
    Result RegisterDownloadSaveDataTransferTask(TransferTaskId* out, DataId id) NN_NOEXCEPT;

    /**
    *       @brief      キャッシュされているデータ情報の個数を取得します。
    *
    *       @return     キャッシュされているデータ情報の個数。
    *
    *       @details    システムにキャッシュされているデータ情報の数を取得します。
    *
    *                   この関数の後に ListDataInfo 等を連続して呼び出す場合、バックグラウンドで意図せずキャッシュが更新され、
    *                   呼び出し間での整合性が崩れる可能性があることに注意してください。
    *                   DaemonController::StopAutonomyTaskExecution を使用することでバックグラウンドでのキャッシュ更新処理を停止することができます。
    *
    */
    int GetDataInfoCount() NN_NOEXCEPT;

    /**
    *       @brief      キャッシュされているデータ情報の一覧を取得します。
    *
    *       @param[out] outValue    取得したデータ情報の格納先。
    *       @param[in]  count       取得する最大数。
    *       @param[in]  offset      取得を開始する位置。
    *
    *       @return     取得した数。
    *
    *       @pre
    *           -   count >= 0
    *           -   outValue のサイズ >= count
    *           -   offset >= 0
    *
    *       @details    キャッシュされているデータ情報の一覧を取得します。
    *
    *                   この API はシステムにキャッシュされているデータ情報を返します。
    *                   サーバの最新の状況を返さないことに注意してください。
    *                   最新の状況が必要な場合は、事前に RequestUpdateDataInfoCache を呼び出し、
    *                   システムにキャッシュされているデータを更新するようにして下さい。
    *
    *                   この API を複数回呼び出すようなケースでは呼び出しの間に
    *                   バックグラウンドで意図せずキャッシュが更新され、呼び出し間での整合性が崩れる可能性があります。
    *                   DaemonController::StopAutonomyTaskExecution を使用することでバックグラウンドでのキャッシュ更新処理を停止することができます。
    *
    */
    int ListDataInfo(DataInfo outValue[], int count, int offset) NN_NOEXCEPT;

    /**
    *       @brief      キャッシュされているデータ情報をアプリケーションを指定して取得します。
    *
    *       @param[out] outValue    取得したデータ情報の格納先。
    *       @param[in]  id          対象のアプリケーション。
    *
    *       @return     データ情報が存在する場合は true。
    *
    *       @pre
    *           -   outValue != nullptr
    *
    *       @details    id の値が一致するデータ情報を取得します。
    *
    *                   この API はシステムにキャッシュされているデータ情報を返します。
    *                   サーバの最新の状況を返さないことに注意してください。
    *                   最新の状況が必要な場合は、事前に RequestUpdateDataInfoCache を呼び出し、
    *                   システムにキャッシュされているデータを更新するようにして下さい。
    *
    */
    bool GetDataInfo(DataInfo* outValue, ApplicationId id) NN_NOEXCEPT;

    /**
    *       @brief      データに紐づくキャッシュされているサムネイルを取得します。
    *
    *       @param[out] outValue    取得したサムネイルの格納先。
    *       @param[in]  dataId      対象のデータ情報のID
    *
    *       @return     サムネイルが存在する場合は true。
    *
    *       @pre
    *           -   outValue != nullptr
    *
    *       @details    キャッシュされているサムネイルを取得します。
    *
    *                   この API はシステムにキャッシュされているサムネイルで id に紐づくものを返します。
    *                   サーバの最新の状況を返さないことに注意してください。
    *                   最新の状況が必要な場合は、事前に RequestUpdateThumbnailCache を呼び出し、
    *                   システムにキャッシュされているデータを更新するようにして下さい。
    *
    */
    bool GetThumbnail(Thumbnail* outValue, DataId dataId) NN_NOEXCEPT;

    /**
    *       @brief      指定したデータの新しさを取得します。
    *
    *       @param[in]  dataId          対象のデータ情報のID
    *
    *       @pre
    *           -   dataId で指定したデータキャッシュが存在する。
    *
    *       @return     データの新しさ
    *
    *       @details    指定したデータの新しさを取得します。
    *                   指定した dataId が指すデータが存在しない場合、本関数の呼び出しは失敗します。
    *
    *
    */
    DataNewness GetDataNewness(DataId dataId) const NN_NOEXCEPT;

    void DeleteDataInfo(DataId id) NN_NOEXCEPT;

private:
    friend void OpenRemoteStorageController(RemoteStorageController* out, const account::Uid& uid) NN_NOEXCEPT;
    friend RemoteStorageController OpenRemoteStorageController(const account::Uid& uid) NN_NOEXCEPT;
    friend void swap(RemoteStorageController& lhs, RemoteStorageController& rhs) NN_NOEXCEPT;
    RemoteStorageController(srv::IRemoteStorageController* interface, const account::Uid& uid) NN_NOEXCEPT;

    srv::IRemoteStorageController* m_Interface;
    account::Uid m_User;
};

void swap(RemoteStorageController& lhs, RemoteStorageController& rhs) NN_NOEXCEPT;

}} // namespace nn::olsc
