﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  olsc で使用する基本的な型や値の定義です。
 */

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc {

//------------------------------------------------------------------------------------

typedef Bit64   ThumbnailId;                            //!< サムネイルの識別子です。
const   size_t  ThumbnailDataSize = 256 * 144 * 4;      //!< サムネイル画像のサイズを表します。 横(256) * 縦(144) * (argb 各 8bit)

/**
*   @brief  サムネイルを表す構造体です。
*/
struct Thumbnail
{
    ThumbnailId     id;                         //!<    サムネイルの ID を表します。
    Bit8            data[ThumbnailDataSize];    //!<    サムネイルのデータを表します。
};

//------------------------------------------------------------------------------------

typedef Bit64   DataId;   //!<    データの識別子です。

/**
*   @brief  データの状態を表す列挙体です。
*/
enum class DataStatus : Bit8
{
    Fixed,                      //!<    バックアップが完了し、サーバに存在する状態。
    Uploading,                  //!<    新規アップロード途中。
    Differential_Uploading,     //!<    差分アップロード途中。
};

/**
*   @brief  データの情報を表す構造体です。
*/
struct DataInfo
{
    DataId                      id;             //!<    データの ID。
    ApplicationId               appId;          //!<    アプリケーションを表します。
    account::Uid                uid;            //!<    ユーザを表します。
    DataStatus                  status;         //!<    データの状態を表します。
    int64_t                     size;           //!<    データのサイズを表します。
    time::PosixTime             updatedTime;    //!<    最後にデータが更新された日時を表します。
    time::PosixTime             uploadedTime;   //!<    最後にデータのアップロードが行われた日時を表します。
};

//------------------------------------------------------------------------------------

/**
*   @brief  データのアップロード制限を表す列挙体です。(内容は現状未定)
*/
enum class DataUploadRestriction : Bit8 {
    None,           //!<    制限なし
    Restricted,     //!<    制限されている
};

/**
*   @brief  データのダウンロード制限を表す列挙体です。(内容は現状未定)
*/
enum class DataDownloadRestriction : Bit8 {
    None,                   //!<    制限なし
    OnlyRepairProcess,      //!<    修理工程でのみダウンロード可能
    Restricted,             //!<    制限されている
};

/**
*   @brief  データの転送ポリシーを表す構造体です。
*/
struct DataTransferPolicy {
    DataUploadRestriction    uploadRestriction;   //!<    アップロード時の制限を表します。
    DataDownloadRestriction  downloadRestriction; //!<    ダウンロード時の制限を表します。
};

/**
*   @brief  データの新しさを表す列挙体です。
*/
enum class DataNewness : Bit8 {
    Equal,                      //!<    ローカルのセーブデータと等しいデータ
    Newer,                      //!<    ローカルのセーブデータと比べ新しいデータ
    Older,                      //!<    ローカルのセーブデータと比べ古いデータ
    Discrete,                   //!<    ローカルのセーブデータと新しさを比較できないデータ
};

}} // namespace nn::olsc
